/*
 * Decompiled with CFR 0.152.
 */
package oracle.jdbc.provider.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.net.ssl.SSLContext;
import oracle.jdbc.provider.util.TNSNames;
import oracle.jdbc.provider.util.TlsUtils;

public final class Wallet {
    private static final String TNS_NAMES_FILE = "tnsnames.ora";
    private static final String KEY_STORE_FILE = "keystore.jks";
    private static final String TRUST_STORE_FILE = "truststore.jks";
    private static final String KEY_STORE_TYPE = "JKS";
    private static final String README_FILE = "README";
    private static final Pattern EXPIRY_PATTERN = Pattern.compile("The SSL certificates provided in this wallet will expire on ([\\d\\-\\.: ]+ UTC)\\.");
    private final TNSNames tnsNames;
    private final OffsetDateTime expirationDate;
    private final SSLContext sslContext;
    private static final DateTimeFormatter EXPIRATION_DATE_FORMATTER = new DateTimeFormatterBuilder().appendPattern("yyyy-MM-dd HH:mm:ss").optionalStart().appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).optionalEnd().appendPattern("X").toFormatter(Locale.ENGLISH);

    private Wallet(TNSNames tnsNames, SSLContext sslContext, OffsetDateTime expirationDate) {
        this.tnsNames = tnsNames;
        this.sslContext = sslContext;
        this.expirationDate = expirationDate;
    }

    public SSLContext getSSLContext() {
        return this.sslContext;
    }

    public OffsetDateTime getExpirationDate() {
        return this.expirationDate;
    }

    public String getHighConnectionString() {
        return this.getConnectionString(TNSNames.ConsumerGroup.HIGH);
    }

    public String getMediumConnectionString() {
        return this.getConnectionString(TNSNames.ConsumerGroup.MEDIUM);
    }

    public String getLowConnectionString() {
        return this.getConnectionString(TNSNames.ConsumerGroup.LOW);
    }

    public String getTransactionProcessingConnectionString() {
        return this.getConnectionString(TNSNames.ConsumerGroup.TRANSACTION_PROCESSING);
    }

    public String getTransactionProcessingUrgentConnectionString() {
        return this.getConnectionString(TNSNames.ConsumerGroup.TRANSACTION_PROCESSING_URGENT);
    }

    private String getConnectionString(TNSNames.ConsumerGroup consumerGroup) {
        String connectionString = this.tnsNames.getConnectionString(consumerGroup);
        if (connectionString == null) {
            throw new IllegalStateException("No connection string available for consumer group: " + (Object)((Object)consumerGroup));
        }
        return connectionString;
    }

    public static Wallet unzip(ZipInputStream zipStream, char[] password) {
        TNSNames tnsNames = null;
        KeyStore keyStore = null;
        KeyStore trustStore = null;
        OffsetDateTime expirationDate = null;
        try {
            ZipEntry entry = zipStream.getNextEntry();
            while (entry != null) {
                switch (entry.getName()) {
                    case "tnsnames.ora": {
                        tnsNames = TNSNames.read(zipStream);
                        break;
                    }
                    case "keystore.jks": {
                        keyStore = TlsUtils.loadKeyStore(zipStream, password, KEY_STORE_TYPE, null);
                        break;
                    }
                    case "truststore.jks": {
                        trustStore = TlsUtils.loadKeyStore(zipStream, null, KEY_STORE_TYPE, null);
                        break;
                    }
                    case "README": {
                        expirationDate = Wallet.parseExpirationDateFromReadme(zipStream);
                        break;
                    }
                }
                entry = zipStream.getNextEntry();
            }
        }
        catch (IOException ioException) {
            throw new IllegalStateException("Failed to read wallet ZIP", ioException);
        }
        if (tnsNames == null) {
            throw Wallet.missingFile(TNS_NAMES_FILE);
        }
        if (keyStore == null) {
            throw Wallet.missingFile(KEY_STORE_FILE);
        }
        if (trustStore == null) {
            throw Wallet.missingFile(TRUST_STORE_FILE);
        }
        return new Wallet(tnsNames, TlsUtils.createSSLContext(keyStore, trustStore, password), expirationDate);
    }

    private static IllegalStateException missingFile(String fileName) {
        return new IllegalStateException("Wallet ZIP did not contain: " + fileName);
    }

    private static String findExpirationDateInStream(InputStream inputStream) throws IOException {
        String line;
        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
        while ((line = reader.readLine()) != null) {
            Matcher matcher = EXPIRY_PATTERN.matcher(line);
            if (!matcher.find()) continue;
            return matcher.group(1).trim().replace(" UTC", "Z");
        }
        return null;
    }

    public static OffsetDateTime parseExpirationDateFromReadme(InputStream inputStream) throws IOException {
        if (inputStream == null) {
            return null;
        }
        String expiryDateString = Wallet.findExpirationDateInStream(inputStream);
        if (expiryDateString == null) {
            return null;
        }
        return Wallet.parseOffsetDateTime(expiryDateString);
    }

    private static OffsetDateTime parseOffsetDateTime(String dateTimeString) {
        try {
            return OffsetDateTime.parse(dateTimeString, EXPIRATION_DATE_FORMATTER);
        }
        catch (DateTimeParseException e) {
            return null;
        }
    }
}

