/*
 * Copyright (C) 2014 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.swap;

import java.io.Serializable;
import java.time.LocalDate;
import java.time.temporal.TemporalAdjuster;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.basics.currency.CurrencyAmount;
import com.opengamma.strata.basics.currency.Payment;

/**
 * An exchange of notionals between two counterparties.
 * <p>
 * In most swaps, the notional amount is not exchanged, with only the net difference being exchanged.
 * However, in certain cases, initial, final or intermediate amounts are exchanged.
 * In this case, the notional can be referred to as the principal.
 * <p>
 * This class represents a notional exchange where the amount is known in advance.
 */
@BeanDefinition(builderScope = "private")
public final class NotionalExchange
    implements SwapPaymentEvent, ImmutableBean, Serializable {

  /**
   * The notional exchange payment.
   * <p>
   * This contains the amount to be paid and the date that payment occurs.
   * This date has been adjusted to be a valid business day.
   */
  @PropertyDefinition(validate = "notNull")
  private final Payment payment;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from the amount and date.
   * 
   * @param paymentAmount  the amount of the notional exchange
   * @param paymentDate  the date that the payment is made
   * @return the notional exchange
   */
  public static NotionalExchange of(CurrencyAmount paymentAmount, LocalDate paymentDate) {
    return new NotionalExchange(Payment.of(paymentAmount, paymentDate));
  }

  /**
   * Obtains an instance from the payment.
   * 
   * @param payment  the payment to be made
   * @return the notional exchange
   */
  public static NotionalExchange of(Payment payment) {
    return new NotionalExchange(payment);
  }

  //-------------------------------------------------------------------------
  @Override
  public LocalDate getPaymentDate() {
    return payment.getDate();
  }

  /**
   * Gets the payment amount.
   * 
   * @return the payment amount
   */
  public CurrencyAmount getPaymentAmount() {
    return payment.getValue();
  }

  /**
   * Gets the currency of the event.
   * <p>
   * The currency of the event is the currency of the payment.
   * 
   * @return the currency of the event
   */
  @Override
  public Currency getCurrency() {
    return payment.getCurrency();
  }

  //-------------------------------------------------------------------------
  @Override
  public NotionalExchange adjustPaymentDate(TemporalAdjuster adjuster) {
    LocalDate adjusted = payment.getDate().with(adjuster);
    return of(payment.getValue(), adjusted);
  }

  @Override
  public boolean isKnownAmountAt(LocalDate date) {
    return true;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code NotionalExchange}.
   * @return the meta-bean, not null
   */
  public static NotionalExchange.Meta meta() {
    return NotionalExchange.Meta.INSTANCE;
  }

  static {
    MetaBean.register(NotionalExchange.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private NotionalExchange(
      Payment payment) {
    JodaBeanUtils.notNull(payment, "payment");
    this.payment = payment;
  }

  @Override
  public NotionalExchange.Meta metaBean() {
    return NotionalExchange.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the notional exchange payment.
   * <p>
   * This contains the amount to be paid and the date that payment occurs.
   * This date has been adjusted to be a valid business day.
   * @return the value of the property, not null
   */
  public Payment getPayment() {
    return payment;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      NotionalExchange other = (NotionalExchange) obj;
      return JodaBeanUtils.equal(payment, other.payment);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(payment);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("NotionalExchange{");
    buf.append("payment").append('=').append(JodaBeanUtils.toString(payment));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code NotionalExchange}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code payment} property.
     */
    private final MetaProperty<Payment> payment = DirectMetaProperty.ofImmutable(
        this, "payment", NotionalExchange.class, Payment.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "payment");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          return payment;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends NotionalExchange> builder() {
      return new NotionalExchange.Builder();
    }

    @Override
    public Class<? extends NotionalExchange> beanType() {
      return NotionalExchange.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code payment} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Payment> payment() {
      return payment;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          return ((NotionalExchange) bean).getPayment();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code NotionalExchange}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<NotionalExchange> {

    private Payment payment;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          return payment;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -786681338:  // payment
          this.payment = (Payment) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public NotionalExchange build() {
      return new NotionalExchange(
          payment);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("NotionalExchange.Builder{");
      buf.append("payment").append('=').append(JodaBeanUtils.toString(payment));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
