/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.rate;

import java.io.Serializable;
import java.time.YearMonth;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableValidator;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ImmutableSet;
import com.opengamma.strata.basics.index.Index;
import com.opengamma.strata.basics.index.PriceIndex;
import com.opengamma.strata.basics.index.PriceIndexObservation;
import com.opengamma.strata.collect.ArgChecker;

/**
 * Defines the computation of inflation figures from a price index with interpolation
 * where the start index value is known.
 * <p>
 * A typical application of this rate computation is payments of a capital indexed bond,
 * where the reference start month is the start month of the bond rather than start month
 * of the payment period.
 * <p>
 * A price index is typically published monthly and has a delay before publication.
 * The rate observed by this instance will be based on the specified start index value
 * and two index observations relative to the end month.
 * Linear interpolation based on the number of days of the payment month is used
 * to find the appropriate value.
 */
@BeanDefinition(builderScope = "private")
public final class InflationEndInterpolatedRateComputation
    implements RateComputation, ImmutableBean, Serializable {

  /**
   * The start index value.
   * <p>
   * The published index value of the start month.
   */
  @PropertyDefinition(validate = "ArgChecker.notNegativeOrZero")
  private final double startIndexValue;
  /**
   * The observation at the end.
   * <p>
   * The inflation rate is the ratio between the start index value and the interpolated end observations.
   * The end month is typically three months before the end of the period.
   */
  @PropertyDefinition(validate = "notNull")
  private final PriceIndexObservation endObservation;
  /**
   * The observation for interpolation at the end.
   * <p>
   * The inflation rate is the ratio between the start index value and the interpolated end observations.
   * The month is typically one month after the month of the end observation.
   */
  @PropertyDefinition(validate = "notNull")
  private final PriceIndexObservation endSecondObservation;
  /**
   * The positive weight used when interpolating.
   * <p>
   * Given two price index observations, typically in adjacent months, the weight is used
   * to determine the adjusted index value. The value is given by the formula
   * {@code (weight * price_index_1 + (1 - weight) * price_index_2)}.
   */
  @PropertyDefinition(validate = "ArgChecker.notNegative")
  private final double weight;

  //-------------------------------------------------------------------------
  /**
   * Creates an instance from an index, start index value and reference end month.
   * <p>
   * The second end observations will be one month later than the end month.
   * 
   * @param index  the index
   * @param startIndexValue  the start index value
   * @param referenceEndMonth  the reference end month
   * @param weight  the weight
   * @return the inflation rate computation
   */
  public static InflationEndInterpolatedRateComputation of(
      PriceIndex index,
      double startIndexValue,
      YearMonth referenceEndMonth,
      double weight) {

    return new InflationEndInterpolatedRateComputation(
        startIndexValue,
        PriceIndexObservation.of(index, referenceEndMonth),
        PriceIndexObservation.of(index, referenceEndMonth.plusMonths(1)),
        weight);
  }

  @ImmutableValidator
  private void validate() {
    ArgChecker.isTrue(
        endObservation.getIndex().equals(endSecondObservation.getIndex()), "Both observations must be for the same index");
    ArgChecker.inOrderNotEqual(
        endObservation.getFixingMonth(), endSecondObservation.getFixingMonth(), "endObservation", "endSecondObservation");
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the Price index.
   * 
   * @return the Price index
   */
  public PriceIndex getIndex() {
    return endObservation.getIndex();
  }

  //-------------------------------------------------------------------------
  @Override
  public void collectIndices(ImmutableSet.Builder<Index> builder) {
    builder.add(getIndex());
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code InflationEndInterpolatedRateComputation}.
   * @return the meta-bean, not null
   */
  public static InflationEndInterpolatedRateComputation.Meta meta() {
    return InflationEndInterpolatedRateComputation.Meta.INSTANCE;
  }

  static {
    MetaBean.register(InflationEndInterpolatedRateComputation.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private InflationEndInterpolatedRateComputation(
      double startIndexValue,
      PriceIndexObservation endObservation,
      PriceIndexObservation endSecondObservation,
      double weight) {
    ArgChecker.notNegativeOrZero(startIndexValue, "startIndexValue");
    JodaBeanUtils.notNull(endObservation, "endObservation");
    JodaBeanUtils.notNull(endSecondObservation, "endSecondObservation");
    ArgChecker.notNegative(weight, "weight");
    this.startIndexValue = startIndexValue;
    this.endObservation = endObservation;
    this.endSecondObservation = endSecondObservation;
    this.weight = weight;
    validate();
  }

  @Override
  public InflationEndInterpolatedRateComputation.Meta metaBean() {
    return InflationEndInterpolatedRateComputation.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the start index value.
   * <p>
   * The published index value of the start month.
   * @return the value of the property
   */
  public double getStartIndexValue() {
    return startIndexValue;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the observation at the end.
   * <p>
   * The inflation rate is the ratio between the start index value and the interpolated end observations.
   * The end month is typically three months before the end of the period.
   * @return the value of the property, not null
   */
  public PriceIndexObservation getEndObservation() {
    return endObservation;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the observation for interpolation at the end.
   * <p>
   * The inflation rate is the ratio between the start index value and the interpolated end observations.
   * The month is typically one month after the month of the end observation.
   * @return the value of the property, not null
   */
  public PriceIndexObservation getEndSecondObservation() {
    return endSecondObservation;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the positive weight used when interpolating.
   * <p>
   * Given two price index observations, typically in adjacent months, the weight is used
   * to determine the adjusted index value. The value is given by the formula
   * {@code (weight * price_index_1 + (1 - weight) * price_index_2)}.
   * @return the value of the property
   */
  public double getWeight() {
    return weight;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      InflationEndInterpolatedRateComputation other = (InflationEndInterpolatedRateComputation) obj;
      return JodaBeanUtils.equal(startIndexValue, other.startIndexValue) &&
          JodaBeanUtils.equal(endObservation, other.endObservation) &&
          JodaBeanUtils.equal(endSecondObservation, other.endSecondObservation) &&
          JodaBeanUtils.equal(weight, other.weight);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(startIndexValue);
    hash = hash * 31 + JodaBeanUtils.hashCode(endObservation);
    hash = hash * 31 + JodaBeanUtils.hashCode(endSecondObservation);
    hash = hash * 31 + JodaBeanUtils.hashCode(weight);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("InflationEndInterpolatedRateComputation{");
    buf.append("startIndexValue").append('=').append(JodaBeanUtils.toString(startIndexValue)).append(',').append(' ');
    buf.append("endObservation").append('=').append(JodaBeanUtils.toString(endObservation)).append(',').append(' ');
    buf.append("endSecondObservation").append('=').append(JodaBeanUtils.toString(endSecondObservation)).append(',').append(' ');
    buf.append("weight").append('=').append(JodaBeanUtils.toString(weight));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code InflationEndInterpolatedRateComputation}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code startIndexValue} property.
     */
    private final MetaProperty<Double> startIndexValue = DirectMetaProperty.ofImmutable(
        this, "startIndexValue", InflationEndInterpolatedRateComputation.class, Double.TYPE);
    /**
     * The meta-property for the {@code endObservation} property.
     */
    private final MetaProperty<PriceIndexObservation> endObservation = DirectMetaProperty.ofImmutable(
        this, "endObservation", InflationEndInterpolatedRateComputation.class, PriceIndexObservation.class);
    /**
     * The meta-property for the {@code endSecondObservation} property.
     */
    private final MetaProperty<PriceIndexObservation> endSecondObservation = DirectMetaProperty.ofImmutable(
        this, "endSecondObservation", InflationEndInterpolatedRateComputation.class, PriceIndexObservation.class);
    /**
     * The meta-property for the {@code weight} property.
     */
    private final MetaProperty<Double> weight = DirectMetaProperty.ofImmutable(
        this, "weight", InflationEndInterpolatedRateComputation.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "startIndexValue",
        "endObservation",
        "endSecondObservation",
        "weight");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1656407615:  // startIndexValue
          return startIndexValue;
        case 82210897:  // endObservation
          return endObservation;
        case 1209389949:  // endSecondObservation
          return endSecondObservation;
        case -791592328:  // weight
          return weight;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends InflationEndInterpolatedRateComputation> builder() {
      return new InflationEndInterpolatedRateComputation.Builder();
    }

    @Override
    public Class<? extends InflationEndInterpolatedRateComputation> beanType() {
      return InflationEndInterpolatedRateComputation.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code startIndexValue} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> startIndexValue() {
      return startIndexValue;
    }

    /**
     * The meta-property for the {@code endObservation} property.
     * @return the meta-property, not null
     */
    public MetaProperty<PriceIndexObservation> endObservation() {
      return endObservation;
    }

    /**
     * The meta-property for the {@code endSecondObservation} property.
     * @return the meta-property, not null
     */
    public MetaProperty<PriceIndexObservation> endSecondObservation() {
      return endSecondObservation;
    }

    /**
     * The meta-property for the {@code weight} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> weight() {
      return weight;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -1656407615:  // startIndexValue
          return ((InflationEndInterpolatedRateComputation) bean).getStartIndexValue();
        case 82210897:  // endObservation
          return ((InflationEndInterpolatedRateComputation) bean).getEndObservation();
        case 1209389949:  // endSecondObservation
          return ((InflationEndInterpolatedRateComputation) bean).getEndSecondObservation();
        case -791592328:  // weight
          return ((InflationEndInterpolatedRateComputation) bean).getWeight();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code InflationEndInterpolatedRateComputation}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<InflationEndInterpolatedRateComputation> {

    private double startIndexValue;
    private PriceIndexObservation endObservation;
    private PriceIndexObservation endSecondObservation;
    private double weight;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1656407615:  // startIndexValue
          return startIndexValue;
        case 82210897:  // endObservation
          return endObservation;
        case 1209389949:  // endSecondObservation
          return endSecondObservation;
        case -791592328:  // weight
          return weight;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1656407615:  // startIndexValue
          this.startIndexValue = (Double) newValue;
          break;
        case 82210897:  // endObservation
          this.endObservation = (PriceIndexObservation) newValue;
          break;
        case 1209389949:  // endSecondObservation
          this.endSecondObservation = (PriceIndexObservation) newValue;
          break;
        case -791592328:  // weight
          this.weight = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public InflationEndInterpolatedRateComputation build() {
      return new InflationEndInterpolatedRateComputation(
          startIndexValue,
          endObservation,
          endSecondObservation,
          weight);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("InflationEndInterpolatedRateComputation.Builder{");
      buf.append("startIndexValue").append('=').append(JodaBeanUtils.toString(startIndexValue)).append(',').append(' ');
      buf.append("endObservation").append('=').append(JodaBeanUtils.toString(endObservation)).append(',').append(' ');
      buf.append("endSecondObservation").append('=').append(JodaBeanUtils.toString(endSecondObservation)).append(',').append(' ');
      buf.append("weight").append('=').append(JodaBeanUtils.toString(weight));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
