/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.credit;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.product.credit.type.CdsQuoteConvention;

/**
 * Market quote for a single-name credit default swap (CDS). 
 * <p>
 * The CDS quote convention and quoted value are stored in this class.
 */
@BeanDefinition(builderScope = "private")
public final class CdsQuote
    implements ImmutableBean, Serializable {

  /**
   * The CDS quote convention.
   */
  @PropertyDefinition(validate = "notNull")
  private final CdsQuoteConvention quoteConvention;
  /**
   * The quoted value.
   * <p>
   * This value must be represented in decimal form.
   */
  @PropertyDefinition
  private final double quotedValue;

  //-------------------------------------------------------------------------
  /**
   * Creates an instance.
   * 
   * @param quoteConvention the quote convention
   * @param quotedValue  the quoted value
   * @return the instance
   */
  public static CdsQuote of(CdsQuoteConvention quoteConvention, double quotedValue) {
    return new CdsQuote(quoteConvention, quotedValue);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code CdsQuote}.
   * @return the meta-bean, not null
   */
  public static CdsQuote.Meta meta() {
    return CdsQuote.Meta.INSTANCE;
  }

  static {
    MetaBean.register(CdsQuote.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private CdsQuote(
      CdsQuoteConvention quoteConvention,
      double quotedValue) {
    JodaBeanUtils.notNull(quoteConvention, "quoteConvention");
    this.quoteConvention = quoteConvention;
    this.quotedValue = quotedValue;
  }

  @Override
  public CdsQuote.Meta metaBean() {
    return CdsQuote.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the CDS quote convention.
   * @return the value of the property, not null
   */
  public CdsQuoteConvention getQuoteConvention() {
    return quoteConvention;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the quoted value.
   * <p>
   * This value must be represented in decimal form.
   * @return the value of the property
   */
  public double getQuotedValue() {
    return quotedValue;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      CdsQuote other = (CdsQuote) obj;
      return JodaBeanUtils.equal(quoteConvention, other.quoteConvention) &&
          JodaBeanUtils.equal(quotedValue, other.quotedValue);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(quoteConvention);
    hash = hash * 31 + JodaBeanUtils.hashCode(quotedValue);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("CdsQuote{");
    buf.append("quoteConvention").append('=').append(JodaBeanUtils.toString(quoteConvention)).append(',').append(' ');
    buf.append("quotedValue").append('=').append(JodaBeanUtils.toString(quotedValue));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code CdsQuote}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code quoteConvention} property.
     */
    private final MetaProperty<CdsQuoteConvention> quoteConvention = DirectMetaProperty.ofImmutable(
        this, "quoteConvention", CdsQuote.class, CdsQuoteConvention.class);
    /**
     * The meta-property for the {@code quotedValue} property.
     */
    private final MetaProperty<Double> quotedValue = DirectMetaProperty.ofImmutable(
        this, "quotedValue", CdsQuote.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "quoteConvention",
        "quotedValue");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 2049149709:  // quoteConvention
          return quoteConvention;
        case 1218162921:  // quotedValue
          return quotedValue;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends CdsQuote> builder() {
      return new CdsQuote.Builder();
    }

    @Override
    public Class<? extends CdsQuote> beanType() {
      return CdsQuote.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code quoteConvention} property.
     * @return the meta-property, not null
     */
    public MetaProperty<CdsQuoteConvention> quoteConvention() {
      return quoteConvention;
    }

    /**
     * The meta-property for the {@code quotedValue} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> quotedValue() {
      return quotedValue;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 2049149709:  // quoteConvention
          return ((CdsQuote) bean).getQuoteConvention();
        case 1218162921:  // quotedValue
          return ((CdsQuote) bean).getQuotedValue();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code CdsQuote}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<CdsQuote> {

    private CdsQuoteConvention quoteConvention;
    private double quotedValue;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 2049149709:  // quoteConvention
          return quoteConvention;
        case 1218162921:  // quotedValue
          return quotedValue;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 2049149709:  // quoteConvention
          this.quoteConvention = (CdsQuoteConvention) newValue;
          break;
        case 1218162921:  // quotedValue
          this.quotedValue = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public CdsQuote build() {
      return new CdsQuote(
          quoteConvention,
          quotedValue);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("CdsQuote.Builder{");
      buf.append("quoteConvention").append('=').append(JodaBeanUtils.toString(quoteConvention)).append(',').append(' ');
      buf.append("quotedValue").append('=').append(JodaBeanUtils.toString(quotedValue));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------

}
