/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.param;

import java.io.Serializable;
import java.time.LocalDate;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutablePreBuild;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.date.Tenor;
import com.opengamma.strata.collect.ArgChecker;

/**
 * Parameter metadata based on a date and tenor.
 */
@BeanDefinition(builderScope = "private")
public final class TenorDateParameterMetadata
    implements DatedParameterMetadata, TenoredParameterMetadata, ImmutableBean, Serializable {

  /**
   * The date associated with the parameter.
   * <p>
   * This is the date that is most closely associated with the parameter.
   * The actual parameter is typically a year fraction based on a day count.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final LocalDate date;
  /**
   * The tenor associated with the parameter.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final Tenor tenor;
  /**
   * The label that describes the parameter, defaulted to the tenor.
   */
  @PropertyDefinition(validate = "notEmpty", overrideGet = true)
  private final String label;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance using the tenor.
   * 
   * @param date  the date associated with the parameter
   * @param tenor  the tenor associated with the parameter
   * @return the parameter metadata based on the tenor
   */
  public static TenorDateParameterMetadata of(LocalDate date, Tenor tenor) {
    ArgChecker.notNull(date, "date");
    ArgChecker.notNull(tenor, "tenor");
    return new TenorDateParameterMetadata(date, tenor, tenor.toString());
  }

  /**
   * Obtains an instance using the tenor, specifying the label.
   * 
   * @param date  the date associated with the parameter
   * @param tenor  the tenor associated with the parameter
   * @param label  the label to use
   * @return the parameter metadata based on the tenor
   */
  public static TenorDateParameterMetadata of(LocalDate date, Tenor tenor, String label) {
    return new TenorDateParameterMetadata(date, tenor, label);
  }

  @ImmutablePreBuild
  private static void preBuild(Builder builder) {
    if (builder.label == null && builder.tenor != null) {
      builder.label = builder.tenor.toString();
    }
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the identifier, which is the tenor.
   *
   * @return the tenor
   */
  @Override
  public Tenor getIdentifier() {
    return tenor;
  }

  @Override
  public TenorDateParameterMetadata withTenor(Tenor tenor) {
    if (this.tenor.equals(tenor)) {
      return this;
    }
    if (label.equals(this.tenor.toString())) {
      return TenorDateParameterMetadata.of(date, tenor);
    } else {
      return TenorDateParameterMetadata.of(date, tenor, label);
    }
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code TenorDateParameterMetadata}.
   * @return the meta-bean, not null
   */
  public static TenorDateParameterMetadata.Meta meta() {
    return TenorDateParameterMetadata.Meta.INSTANCE;
  }

  static {
    MetaBean.register(TenorDateParameterMetadata.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private TenorDateParameterMetadata(
      LocalDate date,
      Tenor tenor,
      String label) {
    JodaBeanUtils.notNull(date, "date");
    JodaBeanUtils.notNull(tenor, "tenor");
    JodaBeanUtils.notEmpty(label, "label");
    this.date = date;
    this.tenor = tenor;
    this.label = label;
  }

  @Override
  public TenorDateParameterMetadata.Meta metaBean() {
    return TenorDateParameterMetadata.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the date associated with the parameter.
   * <p>
   * This is the date that is most closely associated with the parameter.
   * The actual parameter is typically a year fraction based on a day count.
   * @return the value of the property, not null
   */
  @Override
  public LocalDate getDate() {
    return date;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the tenor associated with the parameter.
   * @return the value of the property, not null
   */
  @Override
  public Tenor getTenor() {
    return tenor;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the label that describes the parameter, defaulted to the tenor.
   * @return the value of the property, not empty
   */
  @Override
  public String getLabel() {
    return label;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      TenorDateParameterMetadata other = (TenorDateParameterMetadata) obj;
      return JodaBeanUtils.equal(date, other.date) &&
          JodaBeanUtils.equal(tenor, other.tenor) &&
          JodaBeanUtils.equal(label, other.label);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(date);
    hash = hash * 31 + JodaBeanUtils.hashCode(tenor);
    hash = hash * 31 + JodaBeanUtils.hashCode(label);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(128);
    buf.append("TenorDateParameterMetadata{");
    buf.append("date").append('=').append(JodaBeanUtils.toString(date)).append(',').append(' ');
    buf.append("tenor").append('=').append(JodaBeanUtils.toString(tenor)).append(',').append(' ');
    buf.append("label").append('=').append(JodaBeanUtils.toString(label));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code TenorDateParameterMetadata}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code date} property.
     */
    private final MetaProperty<LocalDate> date = DirectMetaProperty.ofImmutable(
        this, "date", TenorDateParameterMetadata.class, LocalDate.class);
    /**
     * The meta-property for the {@code tenor} property.
     */
    private final MetaProperty<Tenor> tenor = DirectMetaProperty.ofImmutable(
        this, "tenor", TenorDateParameterMetadata.class, Tenor.class);
    /**
     * The meta-property for the {@code label} property.
     */
    private final MetaProperty<String> label = DirectMetaProperty.ofImmutable(
        this, "label", TenorDateParameterMetadata.class, String.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "date",
        "tenor",
        "label");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3076014:  // date
          return date;
        case 110246592:  // tenor
          return tenor;
        case 102727412:  // label
          return label;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends TenorDateParameterMetadata> builder() {
      return new TenorDateParameterMetadata.Builder();
    }

    @Override
    public Class<? extends TenorDateParameterMetadata> beanType() {
      return TenorDateParameterMetadata.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code date} property.
     * @return the meta-property, not null
     */
    public MetaProperty<LocalDate> date() {
      return date;
    }

    /**
     * The meta-property for the {@code tenor} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Tenor> tenor() {
      return tenor;
    }

    /**
     * The meta-property for the {@code label} property.
     * @return the meta-property, not null
     */
    public MetaProperty<String> label() {
      return label;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 3076014:  // date
          return ((TenorDateParameterMetadata) bean).getDate();
        case 110246592:  // tenor
          return ((TenorDateParameterMetadata) bean).getTenor();
        case 102727412:  // label
          return ((TenorDateParameterMetadata) bean).getLabel();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code TenorDateParameterMetadata}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<TenorDateParameterMetadata> {

    private LocalDate date;
    private Tenor tenor;
    private String label;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3076014:  // date
          return date;
        case 110246592:  // tenor
          return tenor;
        case 102727412:  // label
          return label;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 3076014:  // date
          this.date = (LocalDate) newValue;
          break;
        case 110246592:  // tenor
          this.tenor = (Tenor) newValue;
          break;
        case 102727412:  // label
          this.label = (String) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public TenorDateParameterMetadata build() {
      preBuild(this);
      return new TenorDateParameterMetadata(
          date,
          tenor,
          label);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(128);
      buf.append("TenorDateParameterMetadata.Builder{");
      buf.append("date").append('=').append(JodaBeanUtils.toString(date)).append(',').append(' ');
      buf.append("tenor").append('=').append(JodaBeanUtils.toString(tenor)).append(',').append(' ');
      buf.append("label").append('=').append(JodaBeanUtils.toString(label));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
