/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.observable;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.StandardId;
import com.opengamma.strata.data.MarketDataId;

/**
 * Identifies the market data for legal entity information.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class LegalEntityInformationId
    implements MarketDataId<LegalEntityInformation>, ImmutableBean, Serializable {

  /**
  * The legal entity identifier.
  */
  @PropertyDefinition(validate = "notNull")
  private final StandardId legalEntityId;

  //-------------------------------------------------------------------------
  /**
  * Obtains an identifier used to find legal entity information.
  *
  * @param legalEntityId  the name
  * @return an identifier
  */
  public static LegalEntityInformationId of(String legalEntityId) {
    return new LegalEntityInformationId(StandardId.parse(legalEntityId));
  }

  /**
  * Obtains an identifier used to find legal entity information.
  *
  * @param legalEntityId  the name
  * @return the identifier
  */
  public static LegalEntityInformationId of(StandardId legalEntityId) {
    return new LegalEntityInformationId(legalEntityId);
  }

  //-------------------------------------------------------------------------
  @Override
  public Class<LegalEntityInformation> getMarketDataType() {
    return LegalEntityInformation.class;
  }

  @Override
  public String toString() {
    return "LegalEntityInformationId:" + legalEntityId;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code LegalEntityInformationId}.
   */
  private static final TypedMetaBean<LegalEntityInformationId> META_BEAN =
      LightMetaBean.of(
          LegalEntityInformationId.class,
          MethodHandles.lookup(),
          new String[] {
              "legalEntityId"},
          new Object[0]);

  /**
   * The meta-bean for {@code LegalEntityInformationId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<LegalEntityInformationId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private LegalEntityInformationId(
      StandardId legalEntityId) {
    JodaBeanUtils.notNull(legalEntityId, "legalEntityId");
    this.legalEntityId = legalEntityId;
  }

  @Override
  public TypedMetaBean<LegalEntityInformationId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the legal entity identifier.
   * @return the value of the property, not null
   */
  public StandardId getLegalEntityId() {
    return legalEntityId;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      LegalEntityInformationId other = (LegalEntityInformationId) obj;
      return JodaBeanUtils.equal(legalEntityId, other.legalEntityId);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(legalEntityId);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------

}
