/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.observable;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.StandardId;

/**
 * Legal entity information.
 * <p>
 * This stores the information on a legal entity. 
 */
@BeanDefinition(builderScope = "private")
public final class LegalEntityInformation
    implements ImmutableBean, Serializable {

  /**
   * The legal entity identifier. 
   */
  @PropertyDefinition(validate = "notNull")
  private final StandardId legalEntityId;
  /**
   * Whether the legal entity has defaulted or not.
   * <p>
   * True if defaulted, false otherwise.
   */
  @PropertyDefinition(validate = "notNull")
  private final boolean defaulted;

  //-------------------------------------------------------------------------
  /**
   * Creates an instance for a legal entity which has defaulted.
   * 
   * @param legalEntityId  the legal entity ID
   * @return the instance
   */
  public static LegalEntityInformation isDefaulted(StandardId legalEntityId) {
    return new LegalEntityInformation(legalEntityId, true);
  }

  /**
   * Creates an instance for a legal entity which has not defaulted.
   * 
   * @param legalEntityId  the legal entity
   * @return the instance
   */
  public static LegalEntityInformation isNotDefaulted(StandardId legalEntityId) {
    return new LegalEntityInformation(legalEntityId, false);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code LegalEntityInformation}.
   * @return the meta-bean, not null
   */
  public static LegalEntityInformation.Meta meta() {
    return LegalEntityInformation.Meta.INSTANCE;
  }

  static {
    MetaBean.register(LegalEntityInformation.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private LegalEntityInformation(
      StandardId legalEntityId,
      boolean defaulted) {
    JodaBeanUtils.notNull(legalEntityId, "legalEntityId");
    JodaBeanUtils.notNull(defaulted, "defaulted");
    this.legalEntityId = legalEntityId;
    this.defaulted = defaulted;
  }

  @Override
  public LegalEntityInformation.Meta metaBean() {
    return LegalEntityInformation.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the legal entity identifier.
   * @return the value of the property, not null
   */
  public StandardId getLegalEntityId() {
    return legalEntityId;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets whether the legal entity has defaulted or not.
   * <p>
   * True if defaulted, false otherwise.
   * @return the value of the property, not null
   */
  public boolean isDefaulted() {
    return defaulted;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      LegalEntityInformation other = (LegalEntityInformation) obj;
      return JodaBeanUtils.equal(legalEntityId, other.legalEntityId) &&
          (defaulted == other.defaulted);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(legalEntityId);
    hash = hash * 31 + JodaBeanUtils.hashCode(defaulted);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("LegalEntityInformation{");
    buf.append("legalEntityId").append('=').append(JodaBeanUtils.toString(legalEntityId)).append(',').append(' ');
    buf.append("defaulted").append('=').append(JodaBeanUtils.toString(defaulted));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code LegalEntityInformation}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code legalEntityId} property.
     */
    private final MetaProperty<StandardId> legalEntityId = DirectMetaProperty.ofImmutable(
        this, "legalEntityId", LegalEntityInformation.class, StandardId.class);
    /**
     * The meta-property for the {@code defaulted} property.
     */
    private final MetaProperty<Boolean> defaulted = DirectMetaProperty.ofImmutable(
        this, "defaulted", LegalEntityInformation.class, Boolean.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "legalEntityId",
        "defaulted");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          return legalEntityId;
        case -1502128480:  // defaulted
          return defaulted;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends LegalEntityInformation> builder() {
      return new LegalEntityInformation.Builder();
    }

    @Override
    public Class<? extends LegalEntityInformation> beanType() {
      return LegalEntityInformation.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code legalEntityId} property.
     * @return the meta-property, not null
     */
    public MetaProperty<StandardId> legalEntityId() {
      return legalEntityId;
    }

    /**
     * The meta-property for the {@code defaulted} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Boolean> defaulted() {
      return defaulted;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          return ((LegalEntityInformation) bean).getLegalEntityId();
        case -1502128480:  // defaulted
          return ((LegalEntityInformation) bean).isDefaulted();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code LegalEntityInformation}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<LegalEntityInformation> {

    private StandardId legalEntityId;
    private boolean defaulted;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          return legalEntityId;
        case -1502128480:  // defaulted
          return defaulted;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 866287159:  // legalEntityId
          this.legalEntityId = (StandardId) newValue;
          break;
        case -1502128480:  // defaulted
          this.defaulted = (Boolean) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public LegalEntityInformation build() {
      return new LegalEntityInformation(
          legalEntityId,
          defaulted);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("LegalEntityInformation.Builder{");
      buf.append("legalEntityId").append('=').append(JodaBeanUtils.toString(legalEntityId)).append(',').append(' ');
      buf.append("defaulted").append('=').append(JodaBeanUtils.toString(defaulted));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------

}
