/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.curve;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.market.ValueType;
import com.opengamma.strata.market.param.ParameterMetadata;

/**
 * Simple parameter metadata containing the x value and type.
 */
@BeanDefinition(builderScope = "private")
public final class SimpleCurveParameterMetadata
    implements ParameterMetadata, ImmutableBean, Serializable {

  /**
   * The type of the x-value.
   */
  @PropertyDefinition(validate = "notNull")
  private final ValueType xValueType;
  /**
   * The x-value.
   */
  @PropertyDefinition
  private final double xValue;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance specifying information about the x-value.
   * 
   * @param xValueType  the x-value type
   * @param xValue  the x-value
   * @return the parameter metadata based on the date and label
   */
  public static SimpleCurveParameterMetadata of(ValueType xValueType, double xValue) {
    return new SimpleCurveParameterMetadata(xValueType, xValue);
  }

  //-------------------------------------------------------------------------
  @Override
  public String getLabel() {
    return xValueType + "=" + xValue;
  }

  @Override
  public String getIdentifier() {
    return getLabel();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SimpleCurveParameterMetadata}.
   * @return the meta-bean, not null
   */
  public static SimpleCurveParameterMetadata.Meta meta() {
    return SimpleCurveParameterMetadata.Meta.INSTANCE;
  }

  static {
    MetaBean.register(SimpleCurveParameterMetadata.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private SimpleCurveParameterMetadata(
      ValueType xValueType,
      double xValue) {
    JodaBeanUtils.notNull(xValueType, "xValueType");
    this.xValueType = xValueType;
    this.xValue = xValue;
  }

  @Override
  public SimpleCurveParameterMetadata.Meta metaBean() {
    return SimpleCurveParameterMetadata.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the type of the x-value.
   * @return the value of the property, not null
   */
  public ValueType getXValueType() {
    return xValueType;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the x-value.
   * @return the value of the property
   */
  public double getXValue() {
    return xValue;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SimpleCurveParameterMetadata other = (SimpleCurveParameterMetadata) obj;
      return JodaBeanUtils.equal(xValueType, other.xValueType) &&
          JodaBeanUtils.equal(xValue, other.xValue);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(xValueType);
    hash = hash * 31 + JodaBeanUtils.hashCode(xValue);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("SimpleCurveParameterMetadata{");
    buf.append("xValueType").append('=').append(JodaBeanUtils.toString(xValueType)).append(',').append(' ');
    buf.append("xValue").append('=').append(JodaBeanUtils.toString(xValue));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code SimpleCurveParameterMetadata}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code xValueType} property.
     */
    private final MetaProperty<ValueType> xValueType = DirectMetaProperty.ofImmutable(
        this, "xValueType", SimpleCurveParameterMetadata.class, ValueType.class);
    /**
     * The meta-property for the {@code xValue} property.
     */
    private final MetaProperty<Double> xValue = DirectMetaProperty.ofImmutable(
        this, "xValue", SimpleCurveParameterMetadata.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "xValueType",
        "xValue");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -868509005:  // xValueType
          return xValueType;
        case -777049127:  // xValue
          return xValue;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends SimpleCurveParameterMetadata> builder() {
      return new SimpleCurveParameterMetadata.Builder();
    }

    @Override
    public Class<? extends SimpleCurveParameterMetadata> beanType() {
      return SimpleCurveParameterMetadata.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code xValueType} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ValueType> xValueType() {
      return xValueType;
    }

    /**
     * The meta-property for the {@code xValue} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> xValue() {
      return xValue;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -868509005:  // xValueType
          return ((SimpleCurveParameterMetadata) bean).getXValueType();
        case -777049127:  // xValue
          return ((SimpleCurveParameterMetadata) bean).getXValue();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SimpleCurveParameterMetadata}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<SimpleCurveParameterMetadata> {

    private ValueType xValueType;
    private double xValue;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -868509005:  // xValueType
          return xValueType;
        case -777049127:  // xValue
          return xValue;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -868509005:  // xValueType
          this.xValueType = (ValueType) newValue;
          break;
        case -777049127:  // xValue
          this.xValue = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public SimpleCurveParameterMetadata build() {
      return new SimpleCurveParameterMetadata(
          xValueType,
          xValue);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("SimpleCurveParameterMetadata.Builder{");
      buf.append("xValueType").append('=').append(JodaBeanUtils.toString(xValueType)).append(',').append(' ');
      buf.append("xValue").append('=').append(JodaBeanUtils.toString(xValue));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
