/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.curve;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.opengamma.strata.collect.array.DoubleArray;
import com.opengamma.strata.market.param.ParameterMetadata;
import com.opengamma.strata.market.param.ParameterPerturbation;
import com.opengamma.strata.market.param.UnitParameterSensitivity;

/**
 * A curve based on a single constant value.
 * <p>
 * This class defines a curve in terms of a single parameter, the constant value.
 * When queried, {@link #yValue(double)} always returns the constant value.
 * The sensitivity is 1 and the first derivative is 0.
 * <p>
 * The curve has one parameter, the value of the constant.
 */
@BeanDefinition(builderScope = "private")
public final class ConstantCurve
    implements Curve, ImmutableBean, Serializable {

  /**
   * Sensitivity does not vary.
   */
  private static final DoubleArray SENSITIVITY = DoubleArray.of(1d);

  /**
   * The curve metadata.
   * <p>
   * The metadata will not normally have parameter metadata.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final CurveMetadata metadata;
  /**
   * The single y-value.
   */
  @PropertyDefinition
  private final double yValue;

  //-------------------------------------------------------------------------
  /**
   * Creates a constant curve with a specific value.
   * 
   * @param name  the curve name
   * @param yValue  the constant y-value
   * @return the curve
   */
  public static ConstantCurve of(String name, double yValue) {
    return of(CurveName.of(name), yValue);
  }

  /**
   * Creates a constant curve with a specific value.
   * 
   * @param name  the curve name
   * @param yValue  the constant y-value
   * @return the curve
   */
  public static ConstantCurve of(CurveName name, double yValue) {
    return new ConstantCurve(DefaultCurveMetadata.of(name), yValue);
  }

  /**
   * Creates a constant curve with a specific value.
   * 
   * @param metadata  the curve metadata
   * @param yValue  the constant y-value
   * @return the curve
   */
  public static ConstantCurve of(CurveMetadata metadata, double yValue) {
    return new ConstantCurve(metadata, yValue);
  }

  //-------------------------------------------------------------------------
  // ensure standard constructor is invoked
  private Object readResolve() {
    return new ConstantCurve(metadata, yValue);
  }

  //-------------------------------------------------------------------------
  @Override
  public int getParameterCount() {
    return 1;
  }

  @Override
  public double getParameter(int parameterIndex) {
    Preconditions.checkElementIndex(parameterIndex, 1);
    return yValue;
  }

  @Override
  public ConstantCurve withParameter(int parameterIndex, double newValue) {
    Preconditions.checkElementIndex(parameterIndex, 1);
    return new ConstantCurve(metadata, newValue);
  }

  @Override
  public ConstantCurve withPerturbation(ParameterPerturbation perturbation) {
    return new ConstantCurve(metadata, perturbation.perturbParameter(0, yValue, getParameterMetadata(0)));
  }

  //-------------------------------------------------------------------------
  @Override
  public double yValue(double x) {
    return yValue;
  }

  @Override
  public UnitParameterSensitivity yValueParameterSensitivity(double x) {
    ImmutableList<ParameterMetadata> paramMeta = ImmutableList.of(getParameterMetadata(0));
    return UnitParameterSensitivity.of(metadata.getCurveName(), paramMeta, SENSITIVITY);
  }

  @Override
  public double firstDerivative(double x) {
    return 0d;
  }

  //-------------------------------------------------------------------------
  @Override
  public ConstantCurve withMetadata(CurveMetadata metadata) {
    return new ConstantCurve(metadata.withParameterMetadata(null), yValue);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ConstantCurve}.
   * @return the meta-bean, not null
   */
  public static ConstantCurve.Meta meta() {
    return ConstantCurve.Meta.INSTANCE;
  }

  static {
    MetaBean.register(ConstantCurve.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private ConstantCurve(
      CurveMetadata metadata,
      double yValue) {
    JodaBeanUtils.notNull(metadata, "metadata");
    this.metadata = metadata;
    this.yValue = yValue;
  }

  @Override
  public ConstantCurve.Meta metaBean() {
    return ConstantCurve.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the curve metadata.
   * <p>
   * The metadata will not normally have parameter metadata.
   * @return the value of the property, not null
   */
  @Override
  public CurveMetadata getMetadata() {
    return metadata;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the single y-value.
   * @return the value of the property
   */
  public double getYValue() {
    return yValue;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ConstantCurve other = (ConstantCurve) obj;
      return JodaBeanUtils.equal(metadata, other.metadata) &&
          JodaBeanUtils.equal(yValue, other.yValue);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(metadata);
    hash = hash * 31 + JodaBeanUtils.hashCode(yValue);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("ConstantCurve{");
    buf.append("metadata").append('=').append(JodaBeanUtils.toString(metadata)).append(',').append(' ');
    buf.append("yValue").append('=').append(JodaBeanUtils.toString(yValue));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code ConstantCurve}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code metadata} property.
     */
    private final MetaProperty<CurveMetadata> metadata = DirectMetaProperty.ofImmutable(
        this, "metadata", ConstantCurve.class, CurveMetadata.class);
    /**
     * The meta-property for the {@code yValue} property.
     */
    private final MetaProperty<Double> yValue = DirectMetaProperty.ofImmutable(
        this, "yValue", ConstantCurve.class, Double.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "metadata",
        "yValue");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -450004177:  // metadata
          return metadata;
        case -748419976:  // yValue
          return yValue;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends ConstantCurve> builder() {
      return new ConstantCurve.Builder();
    }

    @Override
    public Class<? extends ConstantCurve> beanType() {
      return ConstantCurve.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code metadata} property.
     * @return the meta-property, not null
     */
    public MetaProperty<CurveMetadata> metadata() {
      return metadata;
    }

    /**
     * The meta-property for the {@code yValue} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> yValue() {
      return yValue;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -450004177:  // metadata
          return ((ConstantCurve) bean).getMetadata();
        case -748419976:  // yValue
          return ((ConstantCurve) bean).getYValue();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code ConstantCurve}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<ConstantCurve> {

    private CurveMetadata metadata;
    private double yValue;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -450004177:  // metadata
          return metadata;
        case -748419976:  // yValue
          return yValue;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -450004177:  // metadata
          this.metadata = (CurveMetadata) newValue;
          break;
        case -748419976:  // yValue
          this.yValue = (Double) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public ConstantCurve build() {
      return new ConstantCurve(
          metadata,
          yValue);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("ConstantCurve.Builder{");
      buf.append("metadata").append('=').append(JodaBeanUtils.toString(metadata)).append(',').append(' ');
      buf.append("yValue").append('=').append(JodaBeanUtils.toString(yValue));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
