/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.amount;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.basics.currency.CurrencyAmount;
import com.opengamma.strata.basics.currency.FxRateProvider;
import com.opengamma.strata.product.common.PayReceive;
import com.opengamma.strata.product.swap.ResolvedSwapLeg;
import com.opengamma.strata.product.swap.SwapLegType;

/**
 * Represents an amount associated with one leg of a swap.
 * <p>
 * The amount is supplemented with details which would typically identify the swap leg.
 * These are useful in order to interpret the amount without reference to the full product.
 */
@BeanDefinition
public final class SwapLegAmount
    implements LegAmount, ImmutableBean, Serializable {

  /**
   * The amount associated with the leg.
   * The meaning associated with this amount is implied by the context.
   * This amount may have been currency converted.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final CurrencyAmount amount;
  /**
   * Whether the leg is pay or receive.
   */
  @PropertyDefinition(validate = "notNull")
  private final PayReceive payReceive;
  /**
   * The type of the leg, such as Fixed or Ibor.
   */
  @PropertyDefinition(validate = "notNull")
  private final SwapLegType type;
  /**
   * The currency of the leg.
   * This is not be affected by any currency conversion applied to the amount.
   */
  @PropertyDefinition(validate = "notNull")
  private final Currency currency;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from a swap leg and amount.
   * 
   * @param leg  the swap leg
   * @param amount  the amount
   * @return the swap leg amount
   */
  public static SwapLegAmount of(ResolvedSwapLeg leg, CurrencyAmount amount) {
    return builder()
        .amount(amount)
        .payReceive(leg.getPayReceive())
        .type(leg.getType())
        .currency(leg.getCurrency())
        .build();
  }

  //-------------------------------------------------------------------------
  @Override
  public SwapLegAmount convertedTo(Currency resultCurrency, FxRateProvider rateProvider) {
    CurrencyAmount convertedAmount = amount.convertedTo(resultCurrency, rateProvider);
    return this.toBuilder().amount(convertedAmount).build();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SwapLegAmount}.
   * @return the meta-bean, not null
   */
  public static SwapLegAmount.Meta meta() {
    return SwapLegAmount.Meta.INSTANCE;
  }

  static {
    MetaBean.register(SwapLegAmount.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static SwapLegAmount.Builder builder() {
    return new SwapLegAmount.Builder();
  }

  private SwapLegAmount(
      CurrencyAmount amount,
      PayReceive payReceive,
      SwapLegType type,
      Currency currency) {
    JodaBeanUtils.notNull(amount, "amount");
    JodaBeanUtils.notNull(payReceive, "payReceive");
    JodaBeanUtils.notNull(type, "type");
    JodaBeanUtils.notNull(currency, "currency");
    this.amount = amount;
    this.payReceive = payReceive;
    this.type = type;
    this.currency = currency;
  }

  @Override
  public SwapLegAmount.Meta metaBean() {
    return SwapLegAmount.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the amount associated with the leg.
   * The meaning associated with this amount is implied by the context.
   * This amount may have been currency converted.
   * @return the value of the property, not null
   */
  @Override
  public CurrencyAmount getAmount() {
    return amount;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets whether the leg is pay or receive.
   * @return the value of the property, not null
   */
  public PayReceive getPayReceive() {
    return payReceive;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the type of the leg, such as Fixed or Ibor.
   * @return the value of the property, not null
   */
  public SwapLegType getType() {
    return type;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the currency of the leg.
   * This is not be affected by any currency conversion applied to the amount.
   * @return the value of the property, not null
   */
  public Currency getCurrency() {
    return currency;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SwapLegAmount other = (SwapLegAmount) obj;
      return JodaBeanUtils.equal(amount, other.amount) &&
          JodaBeanUtils.equal(payReceive, other.payReceive) &&
          JodaBeanUtils.equal(type, other.type) &&
          JodaBeanUtils.equal(currency, other.currency);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(amount);
    hash = hash * 31 + JodaBeanUtils.hashCode(payReceive);
    hash = hash * 31 + JodaBeanUtils.hashCode(type);
    hash = hash * 31 + JodaBeanUtils.hashCode(currency);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("SwapLegAmount{");
    buf.append("amount").append('=').append(JodaBeanUtils.toString(amount)).append(',').append(' ');
    buf.append("payReceive").append('=').append(JodaBeanUtils.toString(payReceive)).append(',').append(' ');
    buf.append("type").append('=').append(JodaBeanUtils.toString(type)).append(',').append(' ');
    buf.append("currency").append('=').append(JodaBeanUtils.toString(currency));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code SwapLegAmount}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code amount} property.
     */
    private final MetaProperty<CurrencyAmount> amount = DirectMetaProperty.ofImmutable(
        this, "amount", SwapLegAmount.class, CurrencyAmount.class);
    /**
     * The meta-property for the {@code payReceive} property.
     */
    private final MetaProperty<PayReceive> payReceive = DirectMetaProperty.ofImmutable(
        this, "payReceive", SwapLegAmount.class, PayReceive.class);
    /**
     * The meta-property for the {@code type} property.
     */
    private final MetaProperty<SwapLegType> type = DirectMetaProperty.ofImmutable(
        this, "type", SwapLegAmount.class, SwapLegType.class);
    /**
     * The meta-property for the {@code currency} property.
     */
    private final MetaProperty<Currency> currency = DirectMetaProperty.ofImmutable(
        this, "currency", SwapLegAmount.class, Currency.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "amount",
        "payReceive",
        "type",
        "currency");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1413853096:  // amount
          return amount;
        case -885469925:  // payReceive
          return payReceive;
        case 3575610:  // type
          return type;
        case 575402001:  // currency
          return currency;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public SwapLegAmount.Builder builder() {
      return new SwapLegAmount.Builder();
    }

    @Override
    public Class<? extends SwapLegAmount> beanType() {
      return SwapLegAmount.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code amount} property.
     * @return the meta-property, not null
     */
    public MetaProperty<CurrencyAmount> amount() {
      return amount;
    }

    /**
     * The meta-property for the {@code payReceive} property.
     * @return the meta-property, not null
     */
    public MetaProperty<PayReceive> payReceive() {
      return payReceive;
    }

    /**
     * The meta-property for the {@code type} property.
     * @return the meta-property, not null
     */
    public MetaProperty<SwapLegType> type() {
      return type;
    }

    /**
     * The meta-property for the {@code currency} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Currency> currency() {
      return currency;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -1413853096:  // amount
          return ((SwapLegAmount) bean).getAmount();
        case -885469925:  // payReceive
          return ((SwapLegAmount) bean).getPayReceive();
        case 3575610:  // type
          return ((SwapLegAmount) bean).getType();
        case 575402001:  // currency
          return ((SwapLegAmount) bean).getCurrency();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SwapLegAmount}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<SwapLegAmount> {

    private CurrencyAmount amount;
    private PayReceive payReceive;
    private SwapLegType type;
    private Currency currency;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(SwapLegAmount beanToCopy) {
      this.amount = beanToCopy.getAmount();
      this.payReceive = beanToCopy.getPayReceive();
      this.type = beanToCopy.getType();
      this.currency = beanToCopy.getCurrency();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1413853096:  // amount
          return amount;
        case -885469925:  // payReceive
          return payReceive;
        case 3575610:  // type
          return type;
        case 575402001:  // currency
          return currency;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1413853096:  // amount
          this.amount = (CurrencyAmount) newValue;
          break;
        case -885469925:  // payReceive
          this.payReceive = (PayReceive) newValue;
          break;
        case 3575610:  // type
          this.type = (SwapLegType) newValue;
          break;
        case 575402001:  // currency
          this.currency = (Currency) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public SwapLegAmount build() {
      return new SwapLegAmount(
          amount,
          payReceive,
          type,
          currency);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the amount associated with the leg.
     * The meaning associated with this amount is implied by the context.
     * This amount may have been currency converted.
     * @param amount  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder amount(CurrencyAmount amount) {
      JodaBeanUtils.notNull(amount, "amount");
      this.amount = amount;
      return this;
    }

    /**
     * Sets whether the leg is pay or receive.
     * @param payReceive  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder payReceive(PayReceive payReceive) {
      JodaBeanUtils.notNull(payReceive, "payReceive");
      this.payReceive = payReceive;
      return this;
    }

    /**
     * Sets the type of the leg, such as Fixed or Ibor.
     * @param type  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder type(SwapLegType type) {
      JodaBeanUtils.notNull(type, "type");
      this.type = type;
      return this;
    }

    /**
     * Sets the currency of the leg.
     * This is not be affected by any currency conversion applied to the amount.
     * @param currency  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder currency(Currency currency) {
      JodaBeanUtils.notNull(currency, "currency");
      this.currency = currency;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("SwapLegAmount.Builder{");
      buf.append("amount").append('=').append(JodaBeanUtils.toString(amount)).append(',').append(' ');
      buf.append("payReceive").append('=').append(JodaBeanUtils.toString(payReceive)).append(',').append(' ');
      buf.append("type").append('=').append(JodaBeanUtils.toString(type)).append(',').append(' ');
      buf.append("currency").append('=').append(JodaBeanUtils.toString(currency));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
