/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.cosmosdb.rx.internal;

import com.microsoft.azure.cosmosdb.DocumentClientException;
import com.microsoft.azure.cosmosdb.RetryOptions;
import com.microsoft.azure.cosmosdb.rx.internal.Exceptions;
import com.microsoft.azure.cosmosdb.rx.internal.GlobalEndpointManager;
import com.microsoft.azure.cosmosdb.rx.internal.IDocumentClientRetryPolicy;
import com.microsoft.azure.cosmosdb.rx.internal.IRetryPolicy;
import com.microsoft.azure.cosmosdb.rx.internal.ResourceThrottleRetryPolicy;
import com.microsoft.azure.cosmosdb.rx.internal.RxDocumentServiceRequest;
import com.microsoft.azure.cosmosdb.rx.internal.Utils;
import java.io.IOException;
import java.net.URL;
import java.time.Duration;
import org.apache.commons.collections4.list.UnmodifiableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import rx.Single;

public class ClientRetryPolicy
implements IDocumentClientRetryPolicy {
    private static final Logger logger = LoggerFactory.getLogger(ClientRetryPolicy.class);
    private static final int RetryIntervalInMS = 1000;
    private static final int MaxRetryCount = 120;
    private final IDocumentClientRetryPolicy throttlingRetry;
    private final GlobalEndpointManager globalEndpointManager;
    private final boolean enableEndpointDiscovery;
    private int failoverRetryCount;
    private int sessionTokenRetryCount;
    private boolean isReadRequest;
    private boolean canUseMultipleWriteLocations;
    private URL locationEndpoint;
    private RetryContext retryContext;

    public ClientRetryPolicy(GlobalEndpointManager globalEndpointManager, boolean enableEndpointDiscovery, RetryOptions retryOptions) {
        this.throttlingRetry = new ResourceThrottleRetryPolicy(retryOptions.getMaxRetryAttemptsOnThrottledRequests(), retryOptions.getMaxRetryWaitTimeInSeconds());
        this.globalEndpointManager = globalEndpointManager;
        this.failoverRetryCount = 0;
        this.enableEndpointDiscovery = enableEndpointDiscovery;
        this.sessionTokenRetryCount = 0;
        this.canUseMultipleWriteLocations = false;
    }

    @Override
    public Single<IRetryPolicy.ShouldRetryResult> shouldRetry(Exception e) {
        this.retryContext = null;
        DocumentClientException clientException = (DocumentClientException)((Object)Utils.as((Object)e, DocumentClientException.class));
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 3)) {
            logger.warn("Endpoint not writable. Will refresh cache and retry. {}", (Object)e.toString());
            return this.shouldRetryOnEndpointFailureAsync(false);
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 403) && Exceptions.isSubStatusCode(clientException, 1008) && this.isReadRequest) {
            logger.warn("Endpoint not available for reads. Will refresh cache and retry. {}", (Object)e.toString());
            return this.shouldRetryOnEndpointFailureAsync(true);
        }
        if (clientException != null && clientException.getCause() instanceof IOException || e != null && e instanceof IOException) {
            logger.warn("Endpoint not reachable. Will refresh cache and retry. {}", (Object)e.toString());
            return this.shouldRetryOnEndpointFailureAsync(this.isReadRequest);
        }
        if (clientException != null && Exceptions.isStatusCode(clientException, 404) && Exceptions.isSubStatusCode(clientException, 1002)) {
            return Single.just((Object)this.shouldRetryOnSessionNotAvailable());
        }
        return this.throttlingRetry.shouldRetry(e);
    }

    private IRetryPolicy.ShouldRetryResult shouldRetryOnSessionNotAvailable() {
        ++this.sessionTokenRetryCount;
        if (!this.enableEndpointDiscovery) {
            return IRetryPolicy.ShouldRetryResult.noRetry();
        }
        if (this.canUseMultipleWriteLocations) {
            UnmodifiableList<URL> endpoints;
            UnmodifiableList<URL> unmodifiableList = endpoints = this.isReadRequest ? this.globalEndpointManager.getReadEndpoints() : this.globalEndpointManager.getWriteEndpoints();
            if (this.sessionTokenRetryCount > endpoints.size()) {
                return IRetryPolicy.ShouldRetryResult.noRetry();
            }
            this.retryContext = new RetryContext(this.sessionTokenRetryCount - 1, this.sessionTokenRetryCount > 1, this.sessionTokenRetryCount == endpoints.size());
            return IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO);
        }
        if (this.sessionTokenRetryCount > 1) {
            return IRetryPolicy.ShouldRetryResult.noRetry();
        }
        this.retryContext = new RetryContext(this.sessionTokenRetryCount - 1, false, true);
        return IRetryPolicy.ShouldRetryResult.retryAfter(Duration.ZERO);
    }

    private Single<IRetryPolicy.ShouldRetryResult> shouldRetryOnEndpointFailureAsync(boolean isReadRequest) {
        if (!this.enableEndpointDiscovery || this.failoverRetryCount > 120) {
            logger.warn("ShouldRetryOnEndpointFailureAsync() Not retrying. Retry count = {}", (Object)this.failoverRetryCount);
            return Single.just((Object)IRetryPolicy.ShouldRetryResult.noRetry());
        }
        ++this.failoverRetryCount;
        if (this.isReadRequest) {
            logger.warn("marking the endpoint {} as unavailable for read", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForRead(this.locationEndpoint);
        } else {
            logger.warn("marking the endpoint {} as unavailable for write", (Object)this.locationEndpoint);
            this.globalEndpointManager.markEndpointUnavailableForWrite(this.locationEndpoint);
        }
        Duration retryDelay = Duration.ZERO;
        if (!this.isReadRequest) {
            logger.debug("Failover happening. retryCount {0}", (Object)this.failoverRetryCount);
            if (this.failoverRetryCount > 1) {
                retryDelay = Duration.ofMillis(1000L);
            }
        } else {
            retryDelay = Duration.ofMillis(1000L);
        }
        this.retryContext = new RetryContext(this.failoverRetryCount, false, false);
        return this.globalEndpointManager.refreshLocationAsync(null).andThen(Single.just((Object)IRetryPolicy.ShouldRetryResult.retryAfter(retryDelay)));
    }

    @Override
    public void onBeforeSendRequest(RxDocumentServiceRequest request) {
        this.isReadRequest = request.isReadOnlyRequest();
        this.canUseMultipleWriteLocations = this.globalEndpointManager.CanUseMultipleWriteLocations(request);
        if (request.requestContext != null) {
            request.requestContext.ClearRouteToLocation();
        }
        if (this.retryContext != null) {
            request.requestContext.RouteToLocation(this.retryContext.retryCount, this.retryContext.retryRequestOnPreferredLocations);
            request.clearSessionTokenOnSessionReadFailure = this.retryContext.clearSessionTokenOnSessionNotAvailable;
        }
        this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(request);
        if (request.requestContext != null) {
            request.requestContext.RouteToLocation(this.locationEndpoint);
        }
    }

    private class RetryContext {
        public int retryCount;
        public boolean retryRequestOnPreferredLocations;
        public boolean clearSessionTokenOnSessionNotAvailable;

        public RetryContext(int retryCount, boolean retryRequestOnPreferredLocations, boolean clearSessionTokenOnSessionNotAvailable) {
            this.retryCount = retryCount;
            this.retryRequestOnPreferredLocations = retryRequestOnPreferredLocations;
            this.clearSessionTokenOnSessionNotAvailable = clearSessionTokenOnSessionNotAvailable;
        }
    }
}

