package com.liveperson.messaging.sdk.api;

import android.app.Activity;
import android.app.Notification;
import android.app.PendingIntent;
import android.content.Context;
import android.os.Handler;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.liveperson.api.LivePersonCallback;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.InitLivePersonProperties;
import com.liveperson.infra.PushType;
import com.liveperson.infra.PushUnregisterType;
import com.liveperson.infra.analytics.LPAPIVersion;
import com.liveperson.infra.analytics.support.LPAnalyticsFacadeWrapper;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.callbacks.InitLivePersonCallBack;
import com.liveperson.infra.callbacks.LogoutLivePersonCallBack;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.log.LogLevel;
import com.liveperson.infra.log.LogLine;
import com.liveperson.infra.messaging_ui.MessagingUIFactory;
import com.liveperson.infra.messaging_ui.MessagingUiInitData;
import com.liveperson.infra.messaging_ui.notification.NotificationController;
import com.liveperson.infra.messaging_ui.uicomponents.PushMessageParser;
import com.liveperson.infra.model.PushMessage;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.utils.AnalyticsUtils;
import com.liveperson.infra.utils.SDKUncaughtExceptionHandler;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.controller.ClientProperties;
import com.liveperson.messaging.model.AgentData;
import com.liveperson.messaging.sdk.BuildConfig;
import com.liveperson.messaging.sdk.R;
import com.liveperson.messaging.sdk.api.callbacks.LogoutLivePersonCallback;
import com.liveperson.messaging.sdk.api.callbacks.ShutDownLivePersonCallback;
import com.liveperson.messaging.sdk.api.exceptions.SdkNotInitializedException;
import com.liveperson.messaging.sdk.api.model.ConsumerProfile;
import com.liveperson.monitoring.MonitoringFactory;
import com.liveperson.monitoring.sdk.MonitoringInternalInitParams;

import java.util.List;
import java.util.Map;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000115;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000116;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000119;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000011C;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000011D;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000011E;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000011F;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000120;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000121;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000122;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000123;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000126;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000127;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000128;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000129;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000012A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000012B;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000012C;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000012D;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000130;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000131;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000132;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000133;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000134;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000135;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000136;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000137;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000138;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000139;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000013A;

/**
 * LivePerson Messaging SDK entry point.
 * <p/>
 * You must initialize this class before use. The simplest way is to just do
 * {#code LivePerson.initialize(Context, String)}.
 */
public class LivePerson {

	private static final String TAG = "LivePerson";

	public static final String ACTION_LP_UPDATE_NUM_UNREAD_MESSAGES_ACTION = NotificationController.ACTION_LP_UPDATE_NUM_UNREAD_MESSAGES_ACTION;
	public static final String ACTION_LP_UPDATE_NUM_UNREAD_MESSAGES_EXTRA = NotificationController.ACTION_LP_UPDATE_NUM_UNREAD_MESSAGES_EXTRA;

	private static String mBrandId;

	private static LPAnalyticsFacadeWrapper lpAnalyticsFacadeWrapper;

	/*
	 * Initialize the singleton object of LPAnalyticsFacadeWrapper class :: Needed to start the tracking
	 * */
	static {
		lpAnalyticsFacadeWrapper = LPAnalyticsFacadeWrapper.Companion.getInstance();
	}

	private LivePerson() {
	}
	// Make constructor private - this is a static class that should never be instantiated.

	/**
	 * Initialize the framework
	 *
	 * @param context        Application or activity context
	 * @param initProperties A properties object containing all the necessary
	 *                       information required to initialize the LivePerson SDK.
	 */
	public static void initialize(Context context, final InitLivePersonProperties initProperties) {
		initialize(context, initProperties, LPAPIVersion.VERSION_2);
	}

	/**
	 * Initialize the framework
	 *
	 * @param context        Application or activity context
	 * @param initProperties A properties object containing all the necessary
	 *                       information required to initialize the LivePerson SDK.
	 * @param version        initialize method version to determine which version host app is using.
	 */
	private static void initialize(Context context, final InitLivePersonProperties initProperties, LPAPIVersion version) {
		initializeAnalyticsService(context.getApplicationContext(), initProperties);
		long startTime = AnalyticsUtils.INSTANCE.getStartTime();
		final Context applicationContext = context.getApplicationContext();
		LPLog.INSTANCE.i(TAG, "=== Initializing LivePerson SDK ===");
		LPLog.INSTANCE.i(TAG, ClientProperties.getLoggablePropertyDump(
				applicationContext, "LivePerson Environment Details", initProperties.getAppId()));

		// Setting an UncaughtExceptionHandler, sending the host app's handler to be able to invoke it later.
		Thread.currentThread().setUncaughtExceptionHandler(SDKUncaughtExceptionHandler.getInstance(Thread.currentThread().getUncaughtExceptionHandler()));

		// Save host app callback
		final InitLivePersonCallBack hostAppCallback = initProperties.getInitCallBack();

		InitLivePersonCallBack callbackForMonitoringInit = new InitLivePersonCallBack() {
			@Override
			public void onInitSucceed() {
				LPLog.INSTANCE.d(TAG, "onInitSucceed: lp_messaging_sdk module version = " + BuildConfig.VERSION_NAME);

				// Initialize Monitoring
				if (initProperties.isMonitoringParamsValid()) {
					// If Monitoring already initialized return success
					if (MonitoringFactory.INSTANCE.isInitialized()) {
						LPLog.INSTANCE.d(TAG, "initialize: Monitoring already initialized. Return success");
						lpAnalyticsFacadeWrapper.trackInit(version, startTime, null);
						hostAppCallback.onInitSucceed();
						return;
					}

					LPLog.INSTANCE.d(TAG, "initialize: initializing monitoring");
					// Initialize Monitoring
					if (MonitoringFactory.INSTANCE.initMonitoring(new MonitoringInternalInitParams(applicationContext, mBrandId, initProperties.getMonitoringInitParams().getAppInstallId()))) {
						lpAnalyticsFacadeWrapper.trackInit(version, startTime, null);
						hostAppCallback.onInitSucceed();
					} else {
						lpAnalyticsFacadeWrapper.trackInit(version, startTime, "Monitoring initialization failed");
						hostAppCallback.onInitFailed(new Exception("Monitoring initialization failed"));
					}
				} else {
					lpAnalyticsFacadeWrapper.trackInit(version, startTime, null);
					hostAppCallback.onInitSucceed();
				}
			}

			@Override
			public void onInitFailed(Exception e) {
				lpAnalyticsFacadeWrapper.trackInit(version, startTime, e.toString());
				hostAppCallback.onInitFailed(e);
			}
		};

		initProperties.setInitCallBack(callbackForMonitoringInit);

		//check if initProperties contains all the mandatory params.
		if (!InitLivePersonProperties.isValid(initProperties)) {
			String invalidParamsError = "InitLivePersonProperties not valid or missing parameters.";
			if (hostAppCallback != null) {
				hostAppCallback.onInitFailed(new Exception(invalidParamsError));
			}
			LPLog.INSTANCE.w(TAG, "Invalid InitLivePersonProperties!");
			lpAnalyticsFacadeWrapper.trackInit(version, startTime, invalidParamsError);
			return;
		}
		//try to initialized
		if (!isValidState()) {
			mBrandId = initProperties.getBrandId();

			// Initialize Messaging
			MessagingUIFactory.getInstance().init(applicationContext, new MessagingUiInitData(initProperties, getSDKVersion(), initProperties.getInterceptors()));

		} else {
			lpAnalyticsFacadeWrapper.trackInit(version, startTime, null);
			hostAppCallback.onInitSucceed();
		}
	}

	/**
	 * Choose to enable/disable SDK logs.
	 * If host app sets value to true, LivePerson logs will be enabled otherwise is disabled in release mode.
	 *
	 * @param isDebuggable <b>true</b> to set Logging Level to VERBOSE, <b>false</b> to
	 *                     set Logging Level to ERROR
	 * @deprecated since Nov 8, 2019 - please prefer
	 * `LivePerson.Logging.setSDKLoggingLevel(loggingLevel: LogLevel)`
	 */
	@Deprecated
	public static void setIsDebuggable(boolean isDebuggable) {
		boolean isSDKDebuggable = isDebuggable || BuildConfig.DEBUG;
		lpAnalyticsFacadeWrapper.trackIsDebuggable(mBrandId, isDebuggable);
		LPLog.INSTANCE.setLoggingLevel(isSDKDebuggable ? LogLevel.VERBOSE : LogLevel.ERROR);
	}

	/**
	 * Comprehensive system logging controls for LivePerson.
	 */
	public static class Logging {
		/**
		 * Choose the level at which the LivePerson SDK logs messages to the Android
		 * System Logs (logcat). This is applied immediately and not retroactively, so
		 * modifying this value <i>before</i> initializing the LivePerson SDK is advised.
		 * <br/><br/>
		 * Levels: NONE, ERROR, WARNING, INFO, DEBUG, VERBOSE
		 * <br/><br/>
		 * Log Levels are treated such that each log level includes the levels below /
		 * before itself. For example, if you pick LogLevel.INFO, you will receive INFO
		 * plus WARNING and ERROR level logs.
		 * <br/><br/>
		 * Default Level: INFO in Production, VERBOSE in Debug.
		 *
		 * @param loggingLevel The highest LogLevel at which the MobileSDK should write to
		 *                     the Android System Logs.
		 */
		public static void setSDKLoggingLevel(@NonNull final LogLevel loggingLevel) {
			lpAnalyticsFacadeWrapper.trackSetLogLevel(mBrandId, loggingLevel, false);
			LPLog.INSTANCE.setLoggingLevel(loggingLevel);
		}

		/**
		 * Choose whether or not PII and Authentication details should be prevented from
		 * showing up in the Android System Logs (logcat) and LivePerson log snapshots.
		 * This is applied immediately and not retroactively, so modifying this value
		 * <i>before</i> initializing the LivePerson SDK is advised.
		 * <br/><br/>
		 * Default Value: <b>enabled</b> in Production, <b>disabled</b> in Debug.
		 *
		 * @param enabled <b>true</b> to replace all instances of potential PII in
		 *                future log lines with asterisks ("********"); <b>false</b>
		 *                to print potential PII to the logs in plaintext.
		 */
		public static void setDataMaskingEnabled(final boolean enabled) {
			lpAnalyticsFacadeWrapper.trackSetDataMasking(mBrandId, enabled);
			LPLog.INSTANCE.setDataMaskingEnabled(enabled);
		}

		/**
		 * Gets a snapshot of LivePerson log history, as a List of LogLine objects,
		 * filtered by LogLevel.
		 *
		 * @param filter The highest LogLevel of lines you want included in the snapshot.
		 * @return a List<LogLine> containing all log lines at an equal or lesser logging
		 * level than `filter`, and none of the lines at a greater logging level.
		 */
		public static List<LogLine> getLogSnapshot(LogLevel filter) {
			lpAnalyticsFacadeWrapper.trackGetLogSnapshot(mBrandId, filter);
			return LPLog.INSTANCE.getLogSnapshot(filter);
		}

		/**
		 * Gets a snapshot of log history, as a List of Strings, filtered by LogLevel.
		 *
		 * @param filter The highest LogLevel of lines you want included in the snapshot.
		 * @return a List<String> containing all log lines at an equal or lesser logging
		 * level than `filter`, and none of the lines at a greater logging level.
		 */
		public static List<String> getLogSnapshotStrings(LogLevel filter) {
			lpAnalyticsFacadeWrapper.trackGetLogSnapshotString(mBrandId, filter);
			return LPLog.INSTANCE.getLogSnapshotStrings(filter);
		}

		/**
		 * Gets a snapshot of log history, as a single String with each line separated by
		 * newline characters, filtered by LogLevel.
		 *
		 * @param filter The highest LogLevel of lines you want included in the snapshot.
		 * @return a large String containing all log lines at an equal or lesser logging
		 * level than `filter`, and none of the lines at a greater logging level.
		 */
		public static String getLogSnapshotStringBlock(LogLevel filter) {
			lpAnalyticsFacadeWrapper.trackGetLogSnapshotBlock(mBrandId, filter);
			return LPLog.INSTANCE.getLogSnapshotStringBlock(filter);
		}

		/**
		 * Clears the in-memory log line cache. Lines already written to the System Logs
		 * (logcat) are not affected, only the logs retrievable via getLogSnapshot and
		 * related methods.
		 */
		public static void clearHistory() {
			lpAnalyticsFacadeWrapper.trackLoggingClearHistory(mBrandId);
			LPLog.INSTANCE.clearHistory();
		}
	}

	/**
	 * Shows the conversation screen in Activity Mode.
	 *
	 * @param activity       An activity whose Context will be used to launch the
	 *                       LivePerson Conversation Activity.
	 * @param lpAuthParams   Authentication details for the current user session.
	 * @param convViewParams Behavioral configuration for the Conversation Activity.
	 */
	public static boolean showConversation(Activity activity, LPAuthenticationParams lpAuthParams, ConversationViewParams convViewParams) {
		return showConversation(activity, lpAuthParams, convViewParams, LPAPIVersion.VERSION_3);
	}

	/**
	 * Shows the conversation screen in Activity Mode. Public method to be called from host app.
	 *
	 * @param activity       An activity whose Context will be used to launch the
	 *                       LivePerson Conversation Activity.
	 * @param lpAuthParams   Authentication details for the current user session.
	 * @param convViewParams Behavioral configuration for the Conversation Activity.
	 * @param version        showConversation method version to identify usage by host app
	 */
	private static boolean showConversation(Activity activity, LPAuthenticationParams lpAuthParams, ConversationViewParams convViewParams, LPAPIVersion version) {
		LPLog.INSTANCE.i(TAG, "Showing Conversation Activity");

		if (!isValidState()) {
			String error = "SDK is not initialized. Not showing Conversation Activity.";
			LPLog.INSTANCE.e(TAG, ERR_00000115, error);
			lpAnalyticsFacadeWrapper.trackDisplayConversation(version, lpAuthParams, convViewParams.getCampaignInfo() != null, false, error);
			return false;
		}

		lpAnalyticsFacadeWrapper.trackDisplayConversation(version, lpAuthParams, convViewParams.getCampaignInfo() != null, false, null);
		return MessagingUIFactory.getInstance().showConversation(activity, mBrandId, lpAuthParams, convViewParams);
	}

	/**
	 * Hides the conversation screen, if it was launched in Activity Mode.
	 *
	 * @param activity An Activity whose Context we can send Intents with.
	 */
	@SuppressWarnings({"WeakerAccess", "unused"}) // Clients may still be using this.
	public static void hideConversation(Activity activity) {
		LPLog.INSTANCE.i(TAG, "Hiding Conversation Activity");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000116, "SDK is not initialized. Not hiding Conversation Activity.");
			lpAnalyticsFacadeWrapper.trackHideConversation(LPAPIVersion.VERSION_1, activity, "SDK is not initialized. Not hiding Conversation Activity.");
			return;
		}
		MessagingUIFactory.getInstance().hideConversation(activity);
		lpAnalyticsFacadeWrapper.trackHideConversation(LPAPIVersion.VERSION_1, activity, null);
	}

	/**
	 * Gets the conversation screen as a Fragment for use in your own Activity.
	 *
	 * @param lpAuthParams   Authentication details for the current user session.
	 * @param convViewParams Behavioral configuration for the Conversation Activity.
	 * @return A {@link com.liveperson.infra.messaging_ui.fragment.ConversationFragment}
	 * if everything is ready to go, <b>or null</b> if the SDK is not initialized.
	 */
	@Nullable
	public static Fragment getConversationFragment(LPAuthenticationParams lpAuthParams, ConversationViewParams convViewParams) {
		return getConversationFragment(lpAuthParams, convViewParams, LPAPIVersion.VERSION_3);
	}

	/**
	 * Gets the conversation screen as a Fragment for use in your own Activity.
	 *
	 * @param lpAuthParams   Authentication details for the current user session.
	 * @param convViewParams Behavioral configuration for the Conversation Activity.
	 * @param version        getConversationFragment method version to identify usage by host app
	 * @return A {@link com.liveperson.infra.messaging_ui.fragment.ConversationFragment}
	 * if everything is ready to go, <b>or null</b> if the SDK is not initialized.
	 */
	@Nullable
	private static Fragment getConversationFragment(LPAuthenticationParams lpAuthParams, ConversationViewParams convViewParams, LPAPIVersion version) {
		LPLog.INSTANCE.i(TAG, "Getting Conversation Fragment");

		if (!isValidState()) {
			String error = "SDK is not initialized. Not returning a Conversation Fragment.";
			LPLog.INSTANCE.e(TAG, ERR_00000119, error);
			lpAnalyticsFacadeWrapper.trackDisplayConversation(version, lpAuthParams, convViewParams.getCampaignInfo() != null, true, error);
			return null;
		}
		lpAnalyticsFacadeWrapper.trackDisplayConversation(version, lpAuthParams, convViewParams.getCampaignInfo() != null, true, null);
		return MessagingUIFactory.getInstance().getConversationFragment(mBrandId, lpAuthParams, convViewParams);
	}

	/**
	 * Reconnect with a new authentication key / JWT.
	 *
	 * @param lpAuthenticationParams The new AuthenticationParams to reconnect with.
	 */
	public static void reconnect(LPAuthenticationParams lpAuthenticationParams) {
		LPLog.INSTANCE.i(TAG, "Reconnecting");

		if (!isValidState()) {
			String error = "SDK is not initialized. Cancelling reconnect.";
			lpAnalyticsFacadeWrapper.trackReconnect(LPAPIVersion.VERSION_2, mBrandId, lpAuthenticationParams, error);
			LPLog.INSTANCE.e(TAG, ERR_0000011C, error);
			return;
		}
		lpAnalyticsFacadeWrapper.trackReconnect(LPAPIVersion.VERSION_2, mBrandId, lpAuthenticationParams, null);
		MessagingFactory.getInstance().getController().reconnect(mBrandId, lpAuthenticationParams);
	}

	/**
	 * Register with the LivePerson pusher service. This is how LivePerson's back-end
	 * servers will know where to send Push Notifications for your app.
	 *
	 * @param brandId  Your Brand's ID.
	 * @param appId    Your App's ID.
	 * @param gcmToken A GCM Token that will route Push Notifications to your app.
	 * @deprecated - Use the version with callbacks,
	 * {@link #registerLPPusher(String, String, String, LPAuthenticationParams, ICallback)},
	 * instead.
	 */
	@Deprecated
	public static void registerLPPusher(String brandId, String appId, String gcmToken) {
		LPLog.INSTANCE.i(TAG, "Registering for Push Notifications");
		LPLog.INSTANCE.e(TAG, ERR_0000011D, "Using DEPRECATED method registerLPPusher(String, String, String). " +
				"Use registerLPPusher(String, String, String, LPAuthenticationParams, ICallback) instead.");

		if (!isValidState()) {
			String error = "SDK is not initialized. Cancelling registration.";
			LPLog.INSTANCE.e(TAG, ERR_0000011E, error);
			lpAnalyticsFacadeWrapper.trackRegisterLPPusher(LPAPIVersion.VERSION_1, brandId, PushType.FCM, null, !TextUtils.isEmpty(gcmToken), error);
			return;
		}
		MessagingFactory.getInstance().getController().registerPusher(brandId, appId, gcmToken, PushType.FCM, null, null);
		lpAnalyticsFacadeWrapper.trackRegisterLPPusher(LPAPIVersion.VERSION_1, brandId, PushType.FCM, null, !TextUtils.isEmpty(gcmToken), null);
	}

	/**
	 * Register with the LivePerson pusher service. This is how LivePerson's back-end
	 * servers will know where to send Push Notifications for your app.
	 *
	 * @param brandId                       Your Brand's ID.
	 * @param appId                         Your App's ID.
	 * @param gcmToken                      A GCM Token that will route Push
	 *                                      Notifications to your app.
	 * @param authenticationParams          Authentication info for the current user
	 *                                      session.
	 * @param registrationCompletedCallback A callback that will notify you of either
	 *                                      successful completion, or an Exception
	 *                                      saying why we failed to register.
	 */
	public static void registerLPPusher(String brandId, String appId, String gcmToken, LPAuthenticationParams authenticationParams, final ICallback<Void, Exception> registrationCompletedCallback) {
		LPLog.INSTANCE.i(TAG, "Registering for Push Notifications");

		if (!isValidState()) {
			String error = "SDK is not initialized. Cancelling registration.";
			LPLog.INSTANCE.e(TAG, ERR_0000011F, error);
			lpAnalyticsFacadeWrapper.trackRegisterLPPusher(LPAPIVersion.VERSION_2, brandId, PushType.FCM, authenticationParams, !TextUtils.isEmpty(gcmToken), error);
			return;
		}
		MessagingFactory.getInstance().getController().registerPusher(brandId, appId, gcmToken, PushType.FCM, authenticationParams, registrationCompletedCallback);
		lpAnalyticsFacadeWrapper.trackRegisterLPPusher(LPAPIVersion.VERSION_2, brandId, PushType.FCM, authenticationParams, !TextUtils.isEmpty(gcmToken), null);
	}

	/**
	 * Register with the LivePerson pusher service. This is how LivePerson's back-end
	 * servers will know where to send Push Notifications for your app.
	 *
	 * @param brandId                       Your Brand's ID.
	 * @param appId                         Your App's ID.
	 * @param deviceToken                   A device Token that will route Push
	 *                                      Notifications to your app.
	 * @param pushType                      Push notification type.
	 * @param authenticationParams          Authentication info for the current user
	 *                                      session.
	 * @param registrationCompletedCallback A callback that will notify you of either
	 *                                      successful completion, or an Exception
	 *                                      saying why we failed to register.
	 */
	public static void registerLPPusher(String brandId, String appId, String deviceToken, PushType pushType, LPAuthenticationParams authenticationParams, final ICallback<Void, Exception> registrationCompletedCallback) {
		LPLog.INSTANCE.i(TAG, "Registering for Push Notifications");

		if (!isValidState()) {
			String error = "SDK is not initialized. Cancelling registration.";
			LPLog.INSTANCE.e(TAG, ERR_0000011F, error);
			lpAnalyticsFacadeWrapper.trackRegisterLPPusher(LPAPIVersion.VERSION_3, brandId, pushType, authenticationParams, !TextUtils.isEmpty(deviceToken), error);
			return;
		}
		MessagingFactory.getInstance().getController().registerPusher(brandId, appId, deviceToken, pushType, authenticationParams, registrationCompletedCallback);
		lpAnalyticsFacadeWrapper.trackRegisterLPPusher(LPAPIVersion.VERSION_3, brandId, pushType, authenticationParams, !TextUtils.isEmpty(deviceToken), null);
	}

	/**
	 * Tells the LivePerson SDK to reconnect while in the background. If the SDK is in
	 * the foreground, use {@link #reconnect(LPAuthenticationParams)} instead.
	 *
	 * @param brandId              Your Brand's ID
	 * @param authenticationParams Authentication info for the current user session.
	 */
	public static void updateTokenInBackground(String brandId, LPAuthenticationParams authenticationParams) {
		LPLog.INSTANCE.i(TAG, "Updating LP Token in the Background");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000120, "SDK is not initialized. Cancelling token refresh.");
			lpAnalyticsFacadeWrapper.trackUpdateTokenInBackground(authenticationParams, brandId, "SDK is not initialized. Cancelling token refresh.");
			return;
		}
		lpAnalyticsFacadeWrapper.trackUpdateTokenInBackground(authenticationParams, brandId, null);
		MessagingFactory.getInstance().getController().updateTokenInBackground(brandId, authenticationParams);
	}

	/**
	 * Unregister from the LivePerson pusher service for the given Brand ID. If you are
	 * connected to the internet, this will stop future LivePerson Push Notifications
	 * from reaching your app.
	 *
	 * @param brandId Your Brand's ID
	 * @param appId   Your App's ID
	 * @deprecated - Use {@link #unregisterLPPusher(String, String, ICallback)} instead.
	 */
	@Deprecated
	public static void unregisterLPPusher(String brandId, String appId) {
		LPLog.INSTANCE.i(TAG, "Un-registering for Push Notifications");
		LPLog.INSTANCE.e(TAG, ERR_00000121, "Using DEPRECATED method unregisterLPPusher(String, String). " +
				"Use unregisterLPPusher(String, String, ICallback) instead.");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000122, "SDK is not initialized. Cancelling push un-registration.");
			lpAnalyticsFacadeWrapper.trackUnRegisterPusher(LPAPIVersion.VERSION_1, brandId, !TextUtils.isEmpty(appId), "SDK is not initialized. Cancelling push un-registration.");
			return;
		}
		MessagingFactory.getInstance().getController().unregisterPusher(brandId, appId, null, null, false);
		lpAnalyticsFacadeWrapper.trackUnRegisterPusher(LPAPIVersion.VERSION_1, brandId, !TextUtils.isEmpty(appId), null);
	}

	/**
	 * Unregister from the LivePerson pusher service for the given Brand ID. If you are
	 * connected to the internet, this will stop future LivePerson Push Notifications
	 * from reaching your app.
	 *
	 * @param brandId                        Your Brand's ID
	 * @param appId                          Your App's ID
	 * @param unregisteringCompletedCallback A callback that will notify you of either
	 *                                       successful completion, or an Exception
	 *                                       saying why we failed to unregister.
	 */
	public static void unregisterLPPusher(String brandId, String appId, final ICallback<Void, Exception> unregisteringCompletedCallback) {
		LPLog.INSTANCE.i(TAG, "Un-registering for Push Notifications");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000123, "SDK is not initialized. Cancelling push un-registration.");
			lpAnalyticsFacadeWrapper.trackUnRegisterPusher(LPAPIVersion.VERSION_2, brandId, !TextUtils.isEmpty(appId), "SDK is not initialized. Cancelling push un-registration.");
			return;
		}
		MessagingFactory.getInstance().getController().unregisterPusher(brandId, appId, null, unregisteringCompletedCallback, false);
		lpAnalyticsFacadeWrapper.trackUnRegisterPusher(LPAPIVersion.VERSION_2, brandId, !TextUtils.isEmpty(appId), null);
	}

	/**
	 * Get the LivePerson Messaging SDK version name.
	 *
	 * @return A String containing the current LivePerson MobileSDK version name.
	 */
	@NonNull
	public static String getSDKVersion() {
		LPLog.INSTANCE.i(TAG, "Getting SDK Version Name");
		lpAnalyticsFacadeWrapper.trackGetSDKVersion(mBrandId, BuildConfig.VERSION_NAME);
		return BuildConfig.VERSION_NAME;
	}

	/**
	 * Handle a LivePerson Push Notification.
	 *
	 * @param context          A Context from which we can launch Intents.
	 * @param remoteMessage    A Map containing the Push Notification's RemoteMessage Data.
	 * @param brandId          Your brand's ID
	 * @param showNotification <b>true</b> to show a Notification to the User;
	 *                         <b>false</b> to handle the Push silently.
	 * @return A {@link PushMessage} containing all the data we unpacked from the
	 * RemoteMessage data, <b>or null</b> if we could not unpack the data or if some of
	 * the required data was missing. (Likely scenario - we tried to handle a
	 * non-LivePerson Push.)
	 */
	@Nullable
	public static PushMessage handlePushMessage(Context context, Map<String, String> remoteMessage, String brandId, boolean showNotification) {
		LPLog.INSTANCE.i(TAG, "Handling a new Push Message");

		if (TextUtils.isEmpty(brandId)) {
			String error = "No Brand ID! Ignoring push message.";
			LPLog.INSTANCE.e(TAG, ERR_00000126, error);
			lpAnalyticsFacadeWrapper.trackHandlePushMessage(LPAPIVersion.VERSION_2, context, brandId, showNotification, null, error);
			return null;
		}

		//Parse the bundle in case it's related to LivePerson messages
		PushMessage message = PushMessageParser.parse(brandId, context, remoteMessage);

		if (message != null) {
			NotificationController.instance.addMessageAndDisplayNotification(context, brandId, message, showNotification, R.drawable.liveperson_icon);
		}

		lpAnalyticsFacadeWrapper.trackHandlePushMessage(LPAPIVersion.VERSION_2, context, brandId, showNotification, message, null);
		return message;
	}

	/**
	 * Gets the number of unread messages.
	 * <br/><br/>
	 * Note: SDK needs to be initialized in order to call this API.
	 *
	 * @param brandId Your Brand's ID
	 * @deprecated - Use the asynchronous version,
	 * {@link #getNumUnreadMessages(String, ICallback)}, instead.
	 */
	@Deprecated
	public static int getNumUnreadMessages(String brandId) {
		LPLog.INSTANCE.i(TAG, "Getting the Number of Unread Messages");
		LPLog.INSTANCE.e(TAG, ERR_00000127, "Using DEPRECATED method getNumUnreadMessages(String). " +
				"Use getNumUnreadMessages(String, ICallback) instead.");

		if (TextUtils.isEmpty(brandId)) {
			String error = "No Brand ID! returning -1";
			LPLog.INSTANCE.e(TAG, ERR_00000128, error);
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_1, brandId, false, null, false, error);
			return -1;
		}
		lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_1, brandId, false, null, false, null);
		return NotificationController.instance.getNumUnreadMessages(brandId);
	}


	/**
	 * Gets the number of unread messages.
	 * <br/><br/>
	 * Note: SDK needs to be initialized in order to call this API.
	 *
	 * @param appId    Your App's ID
	 * @param callback A callback that will deliver either a count of unread messages
	 *                 (positive integer), or an Exception saying why getting the unread
	 *                 message count was unsuccessful.
	 * @deprecated - Use {@link #getUnreadMessagesCount(String, LPAuthenticationParams, ICallback)}, instead.
	 */
	public static void getNumUnreadMessages(String appId, final ICallback<Integer, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Getting the Number of Unread Messages");

		if (!isValidState()) {
			String error = "SDK is not initialized. Not getting message counts.";
			LPLog.INSTANCE.e(TAG, ERR_00000129, error);
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_2, mBrandId, !TextUtils.isEmpty(appId), null, true, error);
			callback.onError(new SdkNotInitializedException());
		} else {
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_2, mBrandId, !TextUtils.isEmpty(appId), null, true, null);
			MessagingFactory.getInstance().getController().getNumUnreadMessages(mBrandId, appId, callback);
		}
	}

	/**
	 * Gets the number of unread messages.
	 * <br/><br/>
	 * Note: SDK needs to be initialized in order to call this API.
	 *
	 * @param appId    Your App's ID
	 * @param callback A callback that will deliver either a count of unread messages
	 *                 (positive integer), or an Exception saying why getting the unread
	 *                 message count was unsuccessful.
	 */
	public static void getUnreadMessagesCount(String appId, final ICallback<Integer, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Getting the Unread Message Count");

		if (!isValidState()) {
			String error = "SDK is not initialized. Not getting message counts.";
			LPLog.INSTANCE.e(TAG, ERR_0000012A, error);
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_3, mBrandId, !TextUtils.isEmpty(appId), null, true, error);
			callback.onError(new SdkNotInitializedException());
		} else {
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_3, mBrandId, !TextUtils.isEmpty(appId), null, true, null);
			MessagingFactory.getInstance().getController().getUnreadMessagesCount(mBrandId, appId, null, callback);
		}
	}

	/**
	 * Gets the number of unread messages.
	 * <br/><br/>
	 * Note: SDK needs to be initialized in order to call this API.
	 *
	 * @param appId    Your App's ID
	 * @param callback A callback that will deliver either a count of unread messages
	 *                 (positive integer), or an Exception saying why getting the unread
	 *                 message count was unsuccessful.
	 */
	public static void getUnreadMessagesCount(String appId, LPAuthenticationParams authenticationParams, final ICallback<Integer, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Getting the Unread Message Count");

		if (!isValidState()) {
			String error = "SDK is not initialized. Not getting message counts.";
			LPLog.INSTANCE.e(TAG, ERR_0000012B, error);
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_4, mBrandId, !TextUtils.isEmpty(appId), authenticationParams, true, error);
			callback.onError(new SdkNotInitializedException());
		} else {
			lpAnalyticsFacadeWrapper.trackGetUnreadMessageCount(LPAPIVersion.VERSION_4, mBrandId, !TextUtils.isEmpty(appId), authenticationParams, true, null);
			MessagingFactory.getInstance().getController().getUnreadMessagesCount(mBrandId, appId, authenticationParams, callback);
		}
	}

	/**
	 * Sets the primary LivePerson Callback through which most system events will be
	 * delivered.
	 * <br/><br/>
	 * (Alternatively, you can listen for Local Broadcasts with a
	 * {@link android.content.BroadcastReceiver} - see
	 * <a href="https://developers.liveperson.com/android-callbacks-interface.html#callbacks-interface">LivePerson Callbacks Documentation</a>
	 * for more info.)
	 *
	 * @param listener Your implementation of a {@link LivePersonCallback} listener
	 *                 that will handle events as we publish them.
	 */
	public static void setCallback(final LivePersonCallback listener) {
		LPLog.INSTANCE.i(TAG, "Setting the LivePerson Callback Listener");
		boolean isListenerAvailable = (listener != null) ? true : false;

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_0000012C, "SDK is not initialized. Rejecting callback assignment.");
			lpAnalyticsFacadeWrapper.trackCallback(mBrandId, isListenerAvailable, "SDK is not initialized. Rejecting callback assignment.", true);
			return;
		}
		MessagingFactory.getInstance().getController().setCallback(listener);
		lpAnalyticsFacadeWrapper.trackCallback(mBrandId, isListenerAvailable, null, true);
	}

	/**
	 * Removes the {@link LivePersonCallback} that was set previously with
	 * {@link #setCallback(LivePersonCallback)}.
	 */
	public static void removeCallBack() {
		LPLog.INSTANCE.i(TAG, "Removing the LivePerson Callback Listener");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_0000012D, "SDK is not initialized. Rejecting callback removal.");
			lpAnalyticsFacadeWrapper.trackCallback(mBrandId, false, "SDK is not initialized. Rejecting callback removal.", false);
			return;
		}
		lpAnalyticsFacadeWrapper.trackCallback(mBrandId, false, null, false);
		MessagingFactory.getInstance().getController().removeCallback();
	}

	/**
	 * Sets the User's Profile for this session, relaying that information to the Agent
	 * they are connected with, or will be connected with.
	 *
	 * @param profile A {@link ConsumerProfile} that contains details about the current
	 *                User.
	 */
	public static void setUserProfile(ConsumerProfile profile) {
		LPLog.INSTANCE.i(TAG, "Setting the User's Profile");
		setUserProfile(profile, LPAPIVersion.VERSION_2, false);

	}

	/**
	 * Sets the User's Profile for this session, relaying that information to the Agent
	 * they are connected with, or will be connected with.
	 *
	 * @param profile      The consumerProfile object contains user info
	 * @param lpApiVersion To track which version of setUserProfile method is calling this method
	 * @param hasAppId     For the check whether App Id has been provided
	 *                     use {@link #setUserProfile(ConsumerProfile)}}
	 */

	private static void setUserProfile(ConsumerProfile profile, LPAPIVersion lpApiVersion, boolean hasAppId) {
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000130, "SDK is not initialized. Rejecting User Profile.");
			lpAnalyticsFacadeWrapper.trackSetUserProfile(lpApiVersion, mBrandId, hasAppId,
					!TextUtils.isEmpty(profile.getFirstName()),
					!TextUtils.isEmpty(profile.getLastName()),
					!TextUtils.isEmpty(profile.getPhoneNumber()),
					"SDK is not initialized. Rejecting User Profile.");
			return;
		}
		MessagingFactory.getInstance().getController().sendUserProfile(mBrandId, profile.toUserProfile());
		lpAnalyticsFacadeWrapper.trackSetUserProfile(lpApiVersion, mBrandId, hasAppId,
				!TextUtils.isEmpty(profile.getFirstName()),
				!TextUtils.isEmpty(profile.getLastName()),
				!TextUtils.isEmpty(profile.getPhoneNumber()), null);
	}

	/**
	 * Checks whether there is an open (unresolved) conversation.
	 *
	 * @param callback A callback that will either return a Boolean (<b>true</b> for yes
	 *                 there is an open conversation; <b>false</b> for no open
	 *                 conversations), or an Exception saying why we were unable to
	 *                 determine if any conversations were open or not.
	 */
	public static void checkActiveConversation(final ICallback<Boolean, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Checking for Open Conversations");
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000131, "SDK is not initialized. Not checking Conversations.");
			lpAnalyticsFacadeWrapper.trackCheckActiveConversation(mBrandId, "SDK is not initialized. Not checking Conversations.");
			callback.onError(new SdkNotInitializedException());
		} else {
			MessagingFactory.getInstance().getController().checkActiveConversation(mBrandId, callback);
			lpAnalyticsFacadeWrapper.trackCheckActiveConversation(mBrandId, null);
		}
	}

	/**
	 * Checks whether there is an open (unresolved) conversation that is marked as urgent.
	 *
	 * @param callback A callback that will either return a Boolean (<b>true</b> for yes
	 *                 there is an open and urgent conversation; <b>false</b> for no open
	 *                 conversations or for an open but non-urgent conversation), or an
	 *                 Exception saying why we were unable to determine if any
	 *                 conversations were open or not.
	 */
	public static void checkConversationIsMarkedAsUrgent(final ICallback<Boolean, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Checking for Open & Urgent Conversations");
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000132, "SDK is not initialized. Not checking Urgent Conversations.");
			lpAnalyticsFacadeWrapper.trackCheckConversationIsMarkedAsUrgent(mBrandId, "SDK is not initialized. Not checking Urgent Conversations.");
			callback.onError(new SdkNotInitializedException());
		} else {
			lpAnalyticsFacadeWrapper.trackCheckConversationIsMarkedAsUrgent(mBrandId, null);
			MessagingFactory.getInstance().getController().checkConversationIsMarkedAsUrgent(mBrandId, callback);
		}
	}

	/**
	 * Gets info about the Agent (first name, last name, nickname, employee ID, and
	 * avatar URL) currently assigned to the open Conversation.
	 * <br/><br/>
	 * Useful for getting Agent details after a User taps on an Agent Avatar, expecting
	 * a detail view with the Agent's info.
	 *
	 * @param callback A Callback that will be called when either the Agent's details
	 *                 are ready for use, or a problem was encountered while trying to
	 *                 fetch them.
	 */
	public static void checkAgentID(final ICallback<AgentData, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Fetching Agent Info");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000133, "SDK is not initialized. Not checking Agent details.");
			callback.onError(new SdkNotInitializedException());
			lpAnalyticsFacadeWrapper.trackCheckAgentID(mBrandId, "SDK is not initialized. Not checking Agent details.");
			return;
		}
		MessagingFactory.getInstance().getController().checkAgentID(mBrandId, callback);
		lpAnalyticsFacadeWrapper.trackCheckAgentID(mBrandId, null);
	}

	/**
	 * Marks the current open Conversation as having Urgent priority.
	 */
	public static void markConversationAsUrgent() {
		LPLog.INSTANCE.i(TAG, "Marking Conversation as Urgent Priority");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000134, "SDK is not initialized. Rejecting priority change.");
			lpAnalyticsFacadeWrapper.trackMarkConversation(true, mBrandId, "SDK is not initialized. Rejecting priority change.");
			return;
		}
		MessagingFactory.getInstance().getController().markConversationAsUrgent(mBrandId, mBrandId);
		lpAnalyticsFacadeWrapper.trackMarkConversation(true, mBrandId, null);
	}

	/**
	 * Marks the current open Conversation as having Normal priority.
	 */
	public static void markConversationAsNormal() {
		LPLog.INSTANCE.i(TAG, "Marking Conversation as Normal Priority");
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000135, "SDK is not initialized. Rejecting priority change.");
			lpAnalyticsFacadeWrapper.trackMarkConversation(false, mBrandId, "SDK is not initialized. Rejecting priority change.");
			return;
		}
		lpAnalyticsFacadeWrapper.trackMarkConversation(false, mBrandId, null);
		MessagingFactory.getInstance().getController().markConversationAsNormal(mBrandId, mBrandId);
	}

	/**
	 * Force-resolves the current Conversation the User is having with an Agent.
	 */
	public static void resolveConversation() {
		LPLog.INSTANCE.i(TAG, "Resolving Conversation");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000136, "SDK is not initialized. Rejecting conversation resolution.");
			lpAnalyticsFacadeWrapper.trackResolveConversation(mBrandId, "SDK is not initialized. Rejecting conversation resolution.");
			return;
		}
		MessagingFactory.getInstance().getController().closeCurrentDialog();
		lpAnalyticsFacadeWrapper.trackResolveConversation(mBrandId, null);
	}

	/**
	 * Checks if the LivePerson SDK is initialized properly.
	 *
	 * @return <b>true</b> if the SDK is initialized and the brandId is not empty;
	 * <b>false</b> otherwise.
	 */
	private static boolean isValidState() {
		boolean initialized = MessagingUIFactory.getInstance().isInitialized();
		boolean isEmpty = TextUtils.isEmpty(mBrandId);
		if (initialized && isEmpty) {
			mBrandId = MessagingUIFactory.getInstance().getMessagingUi().getInitBrandId();
		}
		LPLog.INSTANCE.d(TAG, "isInitialized = " + initialized);
		return initialized && !TextUtils.isEmpty(mBrandId);
	}

	/**
	 * Clear all messages and conversations from local storage.
	 * <br/>
	 * This method cannot clear history properly if there is an open and conversation.
	 *
	 * @return <b>true</b> if history was cleared; <b>false</b> if history was not
	 * cleared (due to either an open conversation or no current brand).
	 */
	public static boolean clearHistory() {
		LPLog.INSTANCE.i(TAG, "Clearing Conversation History");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000137, "SDK is not initialized. Rejecting history removal.");
			lpAnalyticsFacadeWrapper.trackClearHistory(mBrandId, "SDK is not initialized. Rejecting history removal.");
			return false;
		}
		lpAnalyticsFacadeWrapper.trackClearHistory(mBrandId, null);
		return MessagingFactory.getInstance().getController().clearHistory(mBrandId);
	}

	/**
	 * De-initializes the LivePerson Messaging SDK <b>without</b> clearing any saved data.
	 * <br/><br/>
	 * This does <b>NOT</b> close the messaging window.
	 * <br/>
	 * To close the Activity, call {@link #hideConversation(Activity)} <b>before</b>
	 * shutdown.
	 * <br/>
	 * To close the Fragment, remove it from its activity's container <b>before</b>
	 * shutdown.
	 */
	public static void shutDown(final ShutDownLivePersonCallback shutdownCallback) {
		LPLog.INSTANCE.i(TAG, "Shutting down LivePerson SDK");
		LPAPIVersion apiVersion = (shutdownCallback == null) ? LPAPIVersion.VERSION_1 : LPAPIVersion.VERSION_2;
		final long startTime = AnalyticsUtils.INSTANCE.getStartTime();
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000138, "SDK is not initialized. Aborting ShutDown Procedure.");
			lpAnalyticsFacadeWrapper.trackShutdown(apiVersion, startTime, mBrandId, "SDK is not initialized. Aborting ShutDown Procedure.");
			return;
		}

		MessagingUIFactory.getInstance().shutDown(new ShutDownCompletionListener() {

			@Override
			public void shutDownCompleted() {
				if (shutdownCallback != null) {
					shutdownCallback.onShutdownSucceed();
				}
				lpAnalyticsFacadeWrapper.trackShutdown(apiVersion, startTime, mBrandId, null);
				mBrandId = null;
			}

			@Override
			public void shutDownFailed() {

				if (shutdownCallback != null) {
					shutdownCallback.onShutdownFailed();
				}
				lpAnalyticsFacadeWrapper.trackShutdown(apiVersion, startTime, mBrandId, "SDK Failed to Shutdown : from shutDownFailed Callback");
			}
		});
	}

	/**
	 * De-initializes the LivePerson Messaging SDK <b>without</b> clearing any saved data.
	 * <br/><br/>
	 * This does <b>NOT</b> close the messaging window.
	 * <br/>
	 * To close the Activity, call {@link #hideConversation(Activity)} <b>before</b>
	 * shutdown.
	 * <br/>
	 * To close the Fragment, remove it from its activity's container <b>before</b>
	 * shutdown.
	 *
	 * @deprecated This does not provide any indication whether the shutdown was succeeded.
	 * Please use {@link #shutDown(ShutDownLivePersonCallback)} instead.
	 */
	@Deprecated
	public static void shutDown() {
		shutDown(null);
	}

	/**
	 * Clears all LivePerson Messaging SDK data and unregisters from push notifications.
	 * <br/><br/>
	 * In order to unregister from push notifications, the device must be connected to
	 * the internet. It will wait for pusher successfully unregistered.
	 * <br/><br/>
	 * This does <b>not</b> close the messaging window. To close the Activity call
	 * {@link #hideConversation(Activity)} <b>before</b> logout.
	 */
	public static void logOut(final Context context, final String brandId, final String appId, final LogoutLivePersonCallback logoutCallback) {
		logOut(context, brandId, appId, false, null, logoutCallback, LPAPIVersion.VERSION_1);
	}

	/**
	 * Clears all LivePerson Messaging SDK data and unregisters from push notifications.
	 * <br/><br/>
	 * In order to unregister from push notifications, the device must be connected to
	 * the internet. It will wait for pusher successfully unregistered.
	 * <br/><br/>
	 * This does <b>not</b> close the messaging window. To close the Activity call
	 * {@link #hideConversation(Activity)} <b>before</b> logout.
	 */
	public static void logOut(final Context context, final String brandId, final String appId, final boolean forceLogOut, final PushUnregisterType type, final LogoutLivePersonCallback logoutCallback) {
		logOut(context, brandId, appId, forceLogOut, type, logoutCallback, LPAPIVersion.VERSION_2);
	}

	/**
	 * Clears all LivePerson Messaging SDK data and unregisters from push notifications.
	 * <br/><br/>
	 * In order to unregister from push notifications, the device must be connected to
	 * the internet.
	 * <br/><br/>
	 * This does <b>not</b> close the messaging window. To close the Activity call
	 * {@link #hideConversation(Activity)} <b>before</b> logout.
	 *
	 * @param context        The Context environment
	 * @param brandId        The Brand's ID
	 * @param appId          Your App's ID
	 * @param forceLogOut    When true, log out will not wait for pusher successfully unregistered. When false, log out will wait for unregister pusher result.
	 *                       Potential issue: If fails to unregister pusher, consumer still can receive push notification.
	 * @param logoutCallback The logout callback.
	 */
	private static void logOut(final Context context, final String brandId, final String appId, final boolean forceLogOut, final PushUnregisterType type,
							   final LogoutLivePersonCallback logoutCallback, final LPAPIVersion lpapiVersion) {
		long startTime = AnalyticsUtils.INSTANCE.getStartTime();
		LPLog.INSTANCE.i(TAG, "Logging out LivePerson SDK and clearing all stored data");

		mBrandId = brandId;
		//Handler to call the host app with the callback on the same thread.
		final Handler logoutHandler = new Handler();
		InitLivePersonProperties initProperties = new InitLivePersonProperties(brandId, appId, null);
		MessagingUiInitData ui = new MessagingUiInitData(initProperties, getSDKVersion(), null);
		MessagingUIFactory.getInstance().logout(context, ui, forceLogOut, type, new LogoutLivePersonCallBack() {

			@Override
			public void onLogoutSucceed() {
				if (MonitoringFactory.INSTANCE.logout()) {
					logoutHandler.post(logoutCallback::onLogoutSucceed);
				} else {
					logoutHandler.post(logoutCallback::onLogoutFailed);
				}
				lpAnalyticsFacadeWrapper.trackLogout(lpapiVersion, brandId, startTime, !TextUtils.isEmpty(appId), forceLogOut, type, null);
			}

			@Override
			public void onLogoutFailed(final Exception e) {
				logoutHandler.post(logoutCallback::onLogoutFailed);
				lpAnalyticsFacadeWrapper.trackLogout(lpapiVersion, brandId, startTime, !TextUtils.isEmpty(appId), forceLogOut, type, e.getMessage());
			}
		});
	}

	/**
	 * Check if device is registered to pusher.
	 *
	 * @param deviceToken            Your FCM token.
	 * @param appId                  Your App's ID.
	 * @param lpAuthenticationParams Authentication Parameters.
	 * @param callback               Notify if current device is registered to pusher.
	 */
	public static void isPusherRegistered(@NonNull String deviceToken, @NonNull String appId, LPAuthenticationParams lpAuthenticationParams, @NonNull ICallback<Boolean, Exception> callback) {
		LPLog.INSTANCE.i(TAG, "Is pusher registered.");

		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_00000139, "SDK is not initialized.");
			lpAnalyticsFacadeWrapper.trackIsPusherRegistered(lpAuthenticationParams.getAuthType(), !TextUtils.isEmpty(deviceToken), mBrandId, "SDK is not initialized");
			return;
		}
		MessagingFactory.getInstance().getController().isPusherRegistered(mBrandId, deviceToken, appId, lpAuthenticationParams, callback);
		lpAnalyticsFacadeWrapper.trackIsPusherRegistered(lpAuthenticationParams.getAuthType(), !TextUtils.isEmpty(deviceToken), mBrandId, null);
	}

	/**
	 * Set a PendingIntent to be used for the Image Upload foreground service
	 * notification's click handling.
	 * <br/>
	 * <b>Note:</b> the foreground service will be used only if the
	 * <i>upload_photo_using_service</i> configuration in <i>branding.xml</i> is set to
	 * <b>true</b>.
	 *
	 * @param pendingIntent A {@link PendingIntent} that will be launched if a User
	 *                      taps on the Notification representing an ongoing image
	 *                      upload operation.
	 */
	public static void setImageServicePendingIntent(PendingIntent pendingIntent) {
		LPLog.INSTANCE.i(TAG, "Setting ImageUploadService PendingIntent");
		if (pendingIntent != null) {
			lpAnalyticsFacadeWrapper.trackSetImageServicePendingIntent(mBrandId, null);
			MessagingFactory.getInstance().getController().setImageServicePendingIntent(pendingIntent);
		}
		lpAnalyticsFacadeWrapper.trackSetImageServicePendingIntent(mBrandId, "setImageServicePendingIntent: Error : Pending Intent is null.");
	}

	/**
	 * Set a Notification Builder to represent the ongoing notification for the Image
	 * Upload foreground service.
	 * <br/><br/>
	 * <b>Note:</b> It is assumed that a pending intent is already added to the given
	 * notification builder, making a call to
	 * {@link #setImageServicePendingIntent(PendingIntent)} unnecessary.
	 * <br/><br/>
	 * <b>Note:</b> the foreground service will be used only if the
	 * <i>upload_photo_using_service</i> configuration in <i>branding.xml</i> is set to
	 * <b>true</b>.
	 *
	 * @param builder A {@link Notification.Builder} that will generate a
	 *                {@link Notification} to represent the Image Upload service.
	 */
	public static void setImageServiceUploadNotificationBuilder(Notification.Builder builder) {
		LPLog.INSTANCE.i(TAG, "Setting ImageUploadService Notification.Builder");
		if (builder != null) {
			lpAnalyticsFacadeWrapper.trackSetImageServiceUploadNotificationBuilder(mBrandId, null);
			MessagingFactory.getInstance().getController().setImageForegroundServiceUploadNotificationBuilder(builder);
		} else {
			lpAnalyticsFacadeWrapper.trackSetImageServiceUploadNotificationBuilder(mBrandId, "setImageServiceUploadNotificationBuilder : Error - Notification.Builder is null.");
		}
	}

	/**
	 * Set a Notification Builder to represent the ongoing notification for the Image
	 * Download foreground service.
	 * <br/><br/>
	 * <b>Note:</b> It is assumed that a pending intent is already added to the given
	 * notification builder.
	 * <br/><br/>
	 * <b>Note:</b> the foreground service will be used only if the
	 * <i>upload_photo_using_service</i> configuration in <i>branding.xml</i> is set to
	 * <b>true</b>.
	 *
	 * @param builder A {@link Notification.Builder} that will generate a
	 *                {@link Notification} to represent the Image Download service.
	 */
	public static void setImageServiceDownloadNotificationBuilder(Notification.Builder builder) {
		LPLog.INSTANCE.i(TAG, "Setting ImageDownloadService Notification.Builder");
		if (builder != null) {
			lpAnalyticsFacadeWrapper.trackSetImageServiceDownloadNotificationBuilder(mBrandId, null);
			MessagingFactory.getInstance().getController().setImageForegroundServiceDownloadNotificationBuilder(builder);
		} else {
			lpAnalyticsFacadeWrapper.trackSetImageServiceDownloadNotificationBuilder(mBrandId, "setImageServiceDownloadNotificationBuilder : Error - Notification.Builder is null.");
		}
	}

	/**
	 * Call this method before call {@link LivePerson#showConversation(Activity, LPAuthenticationParams, ConversationViewParams)} when tapping on push notification.
	 */
	public static void setPushNotificationTapped() {
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_0000013A, "SDK is not initialized. Rejecting push notification setting.");
			lpAnalyticsFacadeWrapper.trackSetPushNotificationTapped(LPAPIVersion.VERSION_1, mBrandId, null, "SDK is not initialized. Rejecting push notification setting.");
			return;
		}
		MessagingFactory.getInstance().setPushNotificationTapped(null);
		lpAnalyticsFacadeWrapper.trackSetPushNotificationTapped(LPAPIVersion.VERSION_1, mBrandId, null, null);
	}

	/**
	 * Call this method before call {@link LivePerson#showConversation(Activity, LPAuthenticationParams, ConversationViewParams)} when tapping on push notification.
	 * @param pushMessageId Notification id obtained from PushMessage object. (Valid only If using LivePerson.handlePush(...))
	 */
	public static void setPushNotificationTapped(String pushMessageId) {
		if (!isValidState()) {
			LPLog.INSTANCE.e(TAG, ERR_0000013A, "SDK is not initialized. Rejecting push notification setting.");
			lpAnalyticsFacadeWrapper.trackSetPushNotificationTapped(LPAPIVersion.VERSION_2, mBrandId, pushMessageId, "SDK is not initialized. Rejecting push notification setting.");
			return;
		}
		MessagingFactory.getInstance().setPushNotificationTapped(pushMessageId);
		lpAnalyticsFacadeWrapper.trackSetPushNotificationTapped(LPAPIVersion.VERSION_2, mBrandId, pushMessageId, null);
	}

	/**
	 * Initialize analytics service to log user events during user's interaction with SDK
	 *
	 * @param applicationContext Host application context
	 * @param initProperties     Brand's initialization properties
	 */
	private static void initializeAnalyticsService(Context applicationContext, InitLivePersonProperties initProperties) {
		MessagingFactory.getInstance().initAnalyticsService(applicationContext, initProperties.getBrandId(), initProperties.getAppId());
	}

	/**
	 * Send analytics data of SDK initialization state. If initialized successfully, error will be empty
	 * @param version method version
	 * @param startTime Time when initialization started
	 * @param error Error If initialization failed, null otherwise
	 */
}
