package com.liveperson.messaging.sdk.api

import androidx.annotation.BoolRes
import com.liveperson.infra.configuration.Configuration
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.messaging.sdk.R

/**
 * Configuration manager to set and query the attribute values of
 * LivePerson Messaging SDK's configurable xml resources.
 * Available attributes are declared on [developer portal](https://developers.liveperson.com/mobile-app-messaging-sdk-for-android-sdk-attributes-5-0-and-above.html).
 *
 * @see set - to setup boolean runtime configuration.
 * @see get - to query boolean value of a specific resource id.
 */
object LPConfig {

    private const val TAG = "LPConfig"

    /**
     * List of available boolean configurations provided by LivePerson Messaging SDK
     * @see [set] method to understand case of user.
     */
    private val booleanVariables by lazy {
        Configuration.scanXmlIdsAndNames(R.bool::class.java)
    }

    /**
     * Sets the runtime boolean configuration value for the provided resource id.
     * Note: Method will set values for only known resource ids which are present within
     * SDK's precompiled set of boolean configurations. List can be found on [developer portal](https://developers.liveperson.com/mobile-app-messaging-sdk-for-android-sdk-attributes-5-0-and-above.html)
     *
     * @param resId - The resource Identifier of a boolean resource to be updated.
     * @param value - New value for the boolean resource.
     */
    @JvmStatic
    fun set(@BoolRes resId: Int, value: Boolean) {
        if (booleanVariables.keys.contains(resId)) {
            Configuration.set(resId, value)
        } else {
            LPLog.e(TAG, ErrorCode.ERR_00000165, "Unknown resource id. Ignore value.")
        }
    }

    /**
     * Return configuration value of an existing boolean resource If available
     * within SDK's precompiled set of boolean configurations.
     *
     * @param resId - The resource Identifier of a boolean resource.
     *
     * @return boolean value of a resource for the provided id
     * or false if boolean resource does not exist in SDK's
     * precompiled set of boolean configurations.
     */
    @JvmStatic
    fun get(@BoolRes resId: Int): Boolean {
        return if (booleanVariables.keys.contains(resId)) {
            Configuration.getBoolean(resId)
        } else {
            LPLog.e(TAG, ErrorCode.ERR_00000166, "Unknown resource id. Value not found.")
            false
        }
    }
}