/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.protocol;

import com.datastax.oss.driver.Assertions;
import com.datastax.oss.driver.internal.core.protocol.ByteBufPrimitiveCodec;
import com.datastax.oss.driver.internal.core.util.ByteBufs;
import com.datastax.oss.protocol.internal.util.Bytes;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import org.assertj.core.api.AbstractThrowableAssert;
import org.junit.Test;

public class ByteBufPrimitiveCodecTest {
    private ByteBufPrimitiveCodec codec = new ByteBufPrimitiveCodec(ByteBufAllocator.DEFAULT);

    @Test
    public void should_concatenate() {
        ByteBuf left = ByteBufs.wrap(202, 254);
        ByteBuf right = ByteBufs.wrap(186, 190);
        Assertions.assertThat(this.codec.concat(left, right)).containsExactly("0xcafebabe");
    }

    @Test
    public void should_concatenate_slices() {
        ByteBuf left = ByteBufs.wrap(0, 202, 254, 0).slice(1, 2);
        ByteBuf right = ByteBufs.wrap(0, 0, 186, 190, 0).slice(2, 2);
        Assertions.assertThat(this.codec.concat(left, right)).containsExactly("0xcafebabe");
    }

    @Test
    public void should_read_inet_v4() {
        ByteBuf source = ByteBufs.wrap(4, 127, 0, 0, 1, 0, 0, 35, 82);
        InetSocketAddress inet = this.codec.readInet((Object)source);
        Assertions.assertThat((String)inet.getAddress().getHostAddress()).isEqualTo("127.0.0.1");
        Assertions.assertThat((int)inet.getPort()).isEqualTo(9042);
    }

    @Test
    public void should_read_inet_v6() {
        ByteBuf lengthAndAddress = ByteBufPrimitiveCodecTest.allocate(17);
        lengthAndAddress.writeByte(16);
        lengthAndAddress.writeLong(0L);
        lengthAndAddress.writeLong(1L);
        ByteBuf source = this.codec.concat(lengthAndAddress, ByteBufs.wrap(0, 0, 35, 82));
        InetSocketAddress inet = this.codec.readInet((Object)source);
        Assertions.assertThat((String)inet.getAddress().getHostAddress()).isEqualTo("0:0:0:0:0:0:0:1");
        Assertions.assertThat((int)inet.getPort()).isEqualTo(9042);
    }

    @Test
    public void should_fail_to_read_inet_if_length_invalid() {
        ByteBuf source = ByteBufs.wrap(3, 127, 0, 1, 0, 0, 35, 82);
        ((AbstractThrowableAssert)org.assertj.core.api.Assertions.assertThatThrownBy(() -> this.codec.readInet((Object)source)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Invalid address length: 3 ([127, 0, 1])");
    }

    @Test
    public void should_read_inetaddr_v4() {
        ByteBuf source = ByteBufs.wrap(4, 127, 0, 0, 1);
        InetAddress inetAddr = this.codec.readInetAddr(source);
        Assertions.assertThat((String)inetAddr.getHostAddress()).isEqualTo("127.0.0.1");
    }

    @Test
    public void should_read_inetaddr_v6() {
        ByteBuf source = ByteBufPrimitiveCodecTest.allocate(17);
        source.writeByte(16);
        source.writeLong(0L);
        source.writeLong(1L);
        InetAddress inetAddr = this.codec.readInetAddr(source);
        Assertions.assertThat((String)inetAddr.getHostAddress()).isEqualTo("0:0:0:0:0:0:0:1");
    }

    @Test
    public void should_fail_to_read_inetaddr_if_length_invalid() {
        ByteBuf source = ByteBufs.wrap(3, 127, 0, 1);
        ((AbstractThrowableAssert)org.assertj.core.api.Assertions.assertThatThrownBy(() -> this.codec.readInetAddr(source)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Invalid address length: 3 ([127, 0, 1])");
    }

    @Test
    public void should_read_bytes() {
        ByteBuf source = ByteBufs.wrap(0, 0, 0, 4, 202, 254, 186, 190);
        ByteBuffer bytes = this.codec.readBytes(source);
        Assertions.assertThat((String)Bytes.toHexString((ByteBuffer)bytes)).isEqualTo("0xcafebabe");
    }

    @Test
    public void should_read_null_bytes() {
        ByteBuf source = ByteBufs.wrap(255, 255, 255, 255);
        Assertions.assertThat((Comparable)this.codec.readBytes(source)).isNull();
    }

    @Test
    public void should_read_short_bytes() {
        ByteBuf source = ByteBufs.wrap(0, 4, 202, 254, 186, 190);
        Assertions.assertThat((String)Bytes.toHexString((byte[])this.codec.readShortBytes(source))).isEqualTo("0xcafebabe");
    }

    @Test
    public void should_read_string() {
        ByteBuf source = ByteBufs.wrap(0, 5, 104, 101, 108, 108, 111);
        Assertions.assertThat((String)this.codec.readString(source)).isEqualTo("hello");
    }

    @Test
    public void should_fail_to_read_string_if_not_enough_characters() {
        ByteBuf source = this.codec.allocate(2);
        source.writeShort(4);
        ((AbstractThrowableAssert)org.assertj.core.api.Assertions.assertThatThrownBy(() -> this.codec.readString(source)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Not enough bytes to read an UTF-8 serialized string of size 4");
    }

    @Test
    public void should_read_long_string() {
        ByteBuf source = ByteBufs.wrap(0, 0, 0, 5, 104, 101, 108, 108, 111);
        Assertions.assertThat((String)this.codec.readLongString(source)).isEqualTo("hello");
    }

    @Test
    public void should_fail_to_read_long_string_if_not_enough_characters() {
        ByteBuf source = this.codec.allocate(4);
        source.writeInt(4);
        ((AbstractThrowableAssert)org.assertj.core.api.Assertions.assertThatThrownBy(() -> this.codec.readLongString(source)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Not enough bytes to read an UTF-8 serialized string of size 4");
    }

    @Test
    public void should_write_inet_v4() throws Exception {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(9);
        InetSocketAddress inet = new InetSocketAddress(InetAddress.getByName("127.0.0.1"), 9042);
        this.codec.writeInet(inet, (Object)dest);
        Assertions.assertThat(dest).containsExactly("0x047f00000100002352");
    }

    @Test
    public void should_write_inet_v6() throws Exception {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(21);
        InetSocketAddress inet = new InetSocketAddress(InetAddress.getByName("::1"), 9042);
        this.codec.writeInet(inet, (Object)dest);
        Assertions.assertThat(dest).containsExactly("0x100000000000000000000000000000000100002352");
    }

    @Test
    public void should_write_inetaddr_v4() throws Exception {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(5);
        InetAddress inetAddr = InetAddress.getByName("127.0.0.1");
        this.codec.writeInetAddr(inetAddr, dest);
        Assertions.assertThat(dest).containsExactly("0x047f000001");
    }

    @Test
    public void should_write_inetaddr_v6() throws Exception {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(17);
        InetAddress inetAddr = InetAddress.getByName("::1");
        this.codec.writeInetAddr(inetAddr, dest);
        Assertions.assertThat(dest).containsExactly("0x1000000000000000000000000000000001");
    }

    @Test
    public void should_write_string() {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(7);
        this.codec.writeString("hello", dest);
        Assertions.assertThat(dest).containsExactly("0x000568656c6c6f");
    }

    @Test
    public void should_write_long_string() {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(9);
        this.codec.writeLongString("hello", dest);
        Assertions.assertThat(dest).containsExactly("0x0000000568656c6c6f");
    }

    @Test
    public void should_write_bytes() {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(8);
        this.codec.writeBytes(Bytes.fromHexString((String)"0xcafebabe"), dest);
        Assertions.assertThat(dest).containsExactly("0x00000004cafebabe");
    }

    @Test
    public void should_write_short_bytes() {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(6);
        this.codec.writeShortBytes(new byte[]{-54, -2, -70, -66}, dest);
        Assertions.assertThat(dest).containsExactly("0x0004cafebabe");
    }

    @Test
    public void should_write_null_bytes() {
        ByteBuf dest = ByteBufPrimitiveCodecTest.allocate(4);
        this.codec.writeBytes((ByteBuffer)null, dest);
        Assertions.assertThat(dest).containsExactly("0xFFFFFFFF");
    }

    private static ByteBuf allocate(int length) {
        return ByteBufAllocator.DEFAULT.buffer(length);
    }
}

