/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.dbclient.impl.http;

import com.databricks.internal.apache.http.HttpResponse;
import com.databricks.internal.apache.http.HttpResponseInterceptor;
import com.databricks.internal.apache.http.client.HttpRequestRetryHandler;
import com.databricks.internal.apache.http.client.protocol.HttpClientContext;
import com.databricks.internal.apache.http.protocol.HttpContext;
import com.databricks.internal.google.common.annotations.VisibleForTesting;
import com.databricks.jdbc.api.internal.IDatabricksConnectionContext;
import com.databricks.jdbc.exception.DatabricksRetryHandlerException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import java.io.IOException;
import java.util.Objects;

public class DatabricksHttpRetryHandler
implements HttpResponseInterceptor,
HttpRequestRetryHandler {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(DatabricksHttpRetryHandler.class);
    static final String RETRY_INTERVAL_KEY = "retryInterval";
    private static final String TEMP_UNAVAILABLE_ACCUMULATED_TIME_KEY = "tempUnavailableAccumulatedTime";
    private static final String RATE_LIMIT_ACCUMULATED_TIME_KEY = "rateLimitAccumulatedTime";
    static final String RETRY_AFTER_HEADER = "Retry-After";
    private static final int DEFAULT_BACKOFF_FACTOR = 2;
    private static final int MIN_BACKOFF_INTERVAL = 1000;
    private static final int MAX_RETRY_INTERVAL = 10000;
    private final IDatabricksConnectionContext connectionContext;

    public DatabricksHttpRetryHandler(IDatabricksConnectionContext connectionContext) {
        this.connectionContext = connectionContext;
    }

    @Override
    public void process(HttpResponse httpResponse, HttpContext httpContext) throws IOException {
        int statusCode = httpResponse.getStatusLine().getStatusCode();
        if (!this.isStatusCodeRetryable(statusCode)) {
            return;
        }
        int retryInterval = -1;
        if (httpResponse.containsHeader(RETRY_AFTER_HEADER)) {
            retryInterval = Integer.parseInt(httpResponse.getFirstHeader(RETRY_AFTER_HEADER).getValue());
        }
        httpContext.setAttribute(RETRY_INTERVAL_KEY, retryInterval);
        DatabricksHttpRetryHandler.initializeRetryAccumulatedTimeIfNotExist(httpContext);
        String errorReason = httpResponse.containsHeader("X-Thriftserver-Error-Message") ? httpResponse.getFirstHeader("X-Thriftserver-Error-Message").getValue() : httpResponse.getStatusLine().getReasonPhrase();
        String errorMessage = String.format("Retry failure. HTTP response code: %s, Error Message: %s", statusCode, errorReason);
        LOGGER.debug(errorMessage);
        throw new DatabricksRetryHandlerException(errorMessage, statusCode);
    }

    @Override
    public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
        int statusCode = DatabricksHttpRetryHandler.getErrorCodeFromException(exception);
        if (!this.isStatusCodeRetryable(statusCode)) {
            return false;
        }
        int retryInterval = (Integer)context.getAttribute(RETRY_INTERVAL_KEY);
        if ((statusCode == 503 || statusCode == 429) && retryInterval == -1) {
            LOGGER.warn("Invalid retry interval in the context " + String.valueOf(context) + " for the error: " + exception.getMessage());
            return false;
        }
        long tempUnavailableAccumulatedTime = DatabricksHttpRetryHandler.getAccumulatedTime(context, TEMP_UNAVAILABLE_ACCUMULATED_TIME_KEY);
        long rateLimitAccumulatedTime = DatabricksHttpRetryHandler.getAccumulatedTime(context, RATE_LIMIT_ACCUMULATED_TIME_KEY);
        if (statusCode == 503 && tempUnavailableAccumulatedTime + (long)retryInterval > (long)this.connectionContext.getTemporarilyUnavailableRetryTimeout()) {
            LOGGER.warn("TemporarilyUnavailableRetry timeout " + this.connectionContext.getTemporarilyUnavailableRetryTimeout() + " has been hit for the error: " + exception.getMessage());
            return false;
        }
        if (statusCode == 429 && rateLimitAccumulatedTime + (long)retryInterval > (long)this.connectionContext.getRateLimitRetryTimeout()) {
            LOGGER.warn("RateLimitRetry timeout " + this.connectionContext.getRateLimitRetryTimeout() + " has been hit for the error: " + exception.getMessage());
            return false;
        }
        boolean isRequestMethodRetryable = DatabricksHttpRetryHandler.isRequestMethodRetryable(((HttpClientContext)context).getRequest().getRequestLine().getMethod());
        if (!isRequestMethodRetryable) {
            return false;
        }
        if (statusCode == 503) {
            context.setAttribute(TEMP_UNAVAILABLE_ACCUMULATED_TIME_KEY, tempUnavailableAccumulatedTime + (long)retryInterval);
        } else if (statusCode == 429) {
            context.setAttribute(RATE_LIMIT_ACCUMULATED_TIME_KEY, rateLimitAccumulatedTime + (long)retryInterval);
        }
        long delayMillis = DatabricksHttpRetryHandler.calculateDelayInMillis(statusCode, executionCount, retryInterval);
        this.doSleepForDelay(delayMillis);
        return true;
    }

    @VisibleForTesting
    static boolean isRequestMethodRetryable(String method) {
        return Objects.equals("GET", method) || Objects.equals("POST", method) || Objects.equals("PUT", method);
    }

    @VisibleForTesting
    static long calculateDelayInMillis(int errorCode, int executionCount, int retryInterval) {
        switch (errorCode) {
            case 429: 
            case 503: {
                return (long)retryInterval * 1000L;
            }
        }
        return DatabricksHttpRetryHandler.calculateExponentialBackoff(executionCount);
    }

    static long calculateExponentialBackoff(int executionCount) {
        return Math.min(1000L * (long)Math.pow(2.0, executionCount), 10000L);
    }

    static int getErrorCodeFromException(IOException exception) {
        if (exception instanceof DatabricksRetryHandlerException) {
            return ((DatabricksRetryHandlerException)exception).getErrCode();
        }
        return 0;
    }

    private static void initializeRetryAccumulatedTimeIfNotExist(HttpContext httpContext) {
        if (httpContext.getAttribute(TEMP_UNAVAILABLE_ACCUMULATED_TIME_KEY) == null) {
            httpContext.setAttribute(TEMP_UNAVAILABLE_ACCUMULATED_TIME_KEY, 0L);
        }
        if (httpContext.getAttribute(RATE_LIMIT_ACCUMULATED_TIME_KEY) == null) {
            httpContext.setAttribute(RATE_LIMIT_ACCUMULATED_TIME_KEY, 0L);
        }
    }

    private static long getAccumulatedTime(HttpContext context, String key) {
        Object value = context.getAttribute(key);
        return value != null ? (Long)value : 0L;
    }

    @VisibleForTesting
    protected void doSleepForDelay(long delayMillis) {
        try {
            Thread.sleep(delayMillis);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Sleep interrupted", e);
        }
    }

    private boolean isStatusCodeRetryable(int statusCode) {
        switch (statusCode) {
            case 503: {
                return this.connectionContext.shouldRetryTemporarilyUnavailableError();
            }
            case 429: {
                return this.connectionContext.shouldRetryRateLimitError();
            }
        }
        return false;
    }
}

