/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.jdbc.api.impl.volume;

import com.databricks.internal.apache.http.HttpEntity;
import com.databricks.internal.apache.http.client.methods.CloseableHttpResponse;
import com.databricks.internal.apache.http.client.methods.HttpDelete;
import com.databricks.internal.apache.http.client.methods.HttpGet;
import com.databricks.internal.apache.http.client.methods.HttpPut;
import com.databricks.internal.apache.http.entity.ContentType;
import com.databricks.internal.apache.http.entity.FileEntity;
import com.databricks.internal.apache.http.entity.InputStreamEntity;
import com.databricks.internal.apache.http.util.EntityUtils;
import com.databricks.internal.google.common.annotations.VisibleForTesting;
import com.databricks.jdbc.api.impl.VolumeOperationStatus;
import com.databricks.jdbc.common.util.HttpUtil;
import com.databricks.jdbc.common.util.VolumeUtil;
import com.databricks.jdbc.dbclient.IDatabricksHttpClient;
import com.databricks.jdbc.exception.DatabricksHttpException;
import com.databricks.jdbc.log.JdbcLogger;
import com.databricks.jdbc.log.JdbcLoggerFactory;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

class VolumeOperationProcessor {
    private static final JdbcLogger LOGGER = JdbcLoggerFactory.getLogger(VolumeOperationProcessor.class);
    private static final String COMMA_SEPARATOR = ",";
    private static final String PARENT_DIRECTORY_REF = "..";
    private static final Long PUT_SIZE_LIMITS = 0x140000000L;
    private final VolumeUtil.VolumeOperationType operationType;
    private final String operationUrl;
    private final String localFilePath;
    private final Map<String, String> headers;
    private final Set<String> allowedVolumeIngestionPaths;
    private final boolean enableVolumeOperations;
    private final boolean isAllowedInputStreamForVolumeOperation;
    private final IDatabricksHttpClient databricksHttpClient;
    private final InputStreamEntity inputStream;
    private final Consumer<HttpEntity> getStreamReceiver;
    private VolumeOperationStatus status;
    private String errorMessage;

    private VolumeOperationProcessor(Builder builder) {
        this.operationType = builder.operationType;
        this.operationUrl = builder.operationUrl;
        this.localFilePath = builder.localFilePath;
        this.headers = builder.headers;
        this.allowedVolumeIngestionPaths = builder.allowedVolumeIngestionPaths;
        this.enableVolumeOperations = builder.enableVolumeOperations;
        this.isAllowedInputStreamForVolumeOperation = builder.isAllowedInputStreamForVolumeOperation;
        this.inputStream = builder.inputStream;
        this.getStreamReceiver = builder.getStreamReceiver;
        this.databricksHttpClient = builder.databricksHttpClient;
        this.status = builder.status;
        this.errorMessage = builder.errorMessage;
    }

    private static Set<String> getAllowedPaths(String allowedVolumeIngestionPathString) {
        if (allowedVolumeIngestionPathString == null || allowedVolumeIngestionPathString.isEmpty()) {
            return Collections.emptySet();
        }
        return new HashSet<String>(Arrays.asList(allowedVolumeIngestionPathString.split(COMMA_SEPARATOR)));
    }

    void process() {
        LOGGER.debug(String.format("Running volume operation {%s} on local file {%s}", new Object[]{this.operationType, this.localFilePath == null ? "" : this.localFilePath}));
        if (this.operationUrl == null || this.operationUrl.isEmpty()) {
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Volume operation URL is not set";
            LOGGER.error(this.errorMessage);
            return;
        }
        this.validateVolumeOperationsOnFileOrStream();
        if (this.status == VolumeOperationStatus.ABORTED) {
            return;
        }
        this.status = VolumeOperationStatus.RUNNING;
        switch (this.operationType) {
            case GET: {
                this.executeGetOperation();
                break;
            }
            case PUT: {
                this.executePutOperation();
                break;
            }
            case REMOVE: {
                this.executeDeleteOperation();
                break;
            }
            default: {
                this.status = VolumeOperationStatus.ABORTED;
                this.errorMessage = "Invalid operation type";
            }
        }
    }

    VolumeOperationStatus getStatus() {
        return this.status;
    }

    String getErrorMessage() {
        return this.errorMessage;
    }

    private void validateVolumeOperationsOnFileOrStream() {
        if (this.isAllowedInputStreamForVolumeOperation) {
            if (!this.enableVolumeOperations) {
                this.status = VolumeOperationStatus.ABORTED;
                this.errorMessage = "enableVolumeOperations property mandatory for Volume operations on stream";
                LOGGER.error(this.errorMessage);
            }
            return;
        }
        if (this.operationType == VolumeUtil.VolumeOperationType.REMOVE) {
            if (!this.enableVolumeOperations && this.allowedVolumeIngestionPaths.isEmpty()) {
                this.status = VolumeOperationStatus.ABORTED;
                this.errorMessage = "enableVolumeOperations property or Volume ingestion paths required for remove operation on Volume";
                LOGGER.error(this.errorMessage);
            }
            return;
        }
        if (this.allowedVolumeIngestionPaths.isEmpty()) {
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Volume ingestion paths are not set";
            LOGGER.error(this.errorMessage);
            return;
        }
        if (this.localFilePath == null || this.localFilePath.isEmpty() || this.localFilePath.contains(PARENT_DIRECTORY_REF)) {
            LOGGER.error("Local file path is invalid {}", this.localFilePath);
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Local file path is invalid";
            return;
        }
        Optional<Boolean> pathMatched = this.allowedVolumeIngestionPaths.stream().map(this.localFilePath::startsWith).filter(x -> x).findFirst();
        if (pathMatched.isEmpty() || !pathMatched.get().booleanValue()) {
            LOGGER.error("Local file path is not allowed {}", this.localFilePath);
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Local file path is not allowed";
        }
    }

    private void closeResponse(CloseableHttpResponse response) {
        if (response != null) {
            try {
                if (response.getEntity() != null) {
                    EntityUtils.consume(response.getEntity());
                }
                response.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void executeGetOperation() {
        block28: {
            HttpGet httpGet = new HttpGet(this.operationUrl);
            this.headers.forEach(httpGet::addHeader);
            CloseableHttpResponse responseStream = null;
            try {
                if (this.isAllowedInputStreamForVolumeOperation) {
                    responseStream = this.databricksHttpClient.execute(httpGet);
                    if (!HttpUtil.isSuccessfulHttpResponse(responseStream)) {
                        this.status = VolumeOperationStatus.FAILED;
                        this.errorMessage = String.format("Failed to fetch content from volume with error code {%s} for input stream and error {%s}", responseStream.getStatusLine().getStatusCode(), responseStream.getStatusLine().getReasonPhrase());
                        LOGGER.error(this.errorMessage);
                        this.closeResponse(responseStream);
                        return;
                    }
                    this.getStreamReceiver.accept(responseStream.getEntity());
                    this.status = VolumeOperationStatus.SUCCEEDED;
                    return;
                }
            }
            catch (DatabricksHttpException e) {
                this.closeResponse(responseStream);
                this.status = VolumeOperationStatus.FAILED;
                this.errorMessage = "Failed to execute GET operation for input stream: " + e.getMessage();
                LOGGER.error(this.errorMessage);
                return;
            }
            File localFile = new File(this.localFilePath);
            if (localFile.exists()) {
                LOGGER.error("Local file already exists for GET operation {}", this.localFilePath);
                this.status = VolumeOperationStatus.ABORTED;
                this.errorMessage = "Local file already exists";
                return;
            }
            try (CloseableHttpResponse response = this.databricksHttpClient.execute(httpGet);){
                if (!HttpUtil.isSuccessfulHttpResponse(response)) {
                    LOGGER.error("Failed to fetch content from volume with error {%s} for local file {%s}", response.getStatusLine().getStatusCode(), this.localFilePath);
                    this.status = VolumeOperationStatus.FAILED;
                    this.errorMessage = "Failed to download file";
                    return;
                }
                HttpEntity entity = response.getEntity();
                if (entity == null) break block28;
                InputStream inputStream = entity.getContent();
                try (FileOutputStream outputStream = new FileOutputStream(localFile);){
                    int length;
                    byte[] buffer = new byte[1024];
                    while ((length = inputStream.read(buffer)) != -1) {
                        outputStream.write(buffer, 0, length);
                    }
                    this.status = VolumeOperationStatus.SUCCEEDED;
                }
                catch (FileNotFoundException e) {
                    LOGGER.error("Local file path is invalid or a directory {}", this.localFilePath);
                    this.status = VolumeOperationStatus.FAILED;
                    this.errorMessage = "Local file path is invalid or a directory";
                }
                catch (IOException e) {
                    LOGGER.error(e, "Failed to write to local file {%s} with error {%s}", this.localFilePath, e.getMessage());
                    this.status = VolumeOperationStatus.FAILED;
                    this.errorMessage = "Failed to write to local file: " + e.getMessage();
                }
                finally {
                    EntityUtils.consume(entity);
                }
            }
            catch (DatabricksHttpException | IOException e) {
                this.status = VolumeOperationStatus.FAILED;
                this.errorMessage = "Failed to download file: " + e.getMessage();
            }
        }
    }

    @VisibleForTesting
    void executePutOperation() {
        HttpPut httpPut = new HttpPut(this.operationUrl);
        this.headers.forEach(httpPut::addHeader);
        if (this.isAllowedInputStreamForVolumeOperation) {
            if (this.inputStream == null) {
                this.status = VolumeOperationStatus.ABORTED;
                this.errorMessage = "InputStream not set for PUT operation";
                LOGGER.error(this.errorMessage);
                return;
            }
            httpPut.setEntity(this.inputStream);
        } else {
            File file = new File(this.localFilePath);
            if (this.localFileHasErrorForPutOperation(file)) {
                return;
            }
            httpPut.setEntity(new FileEntity(file, ContentType.DEFAULT_BINARY));
        }
        try (CloseableHttpResponse response = this.databricksHttpClient.execute(httpPut);){
            if (HttpUtil.isSuccessfulHttpResponse(response)) {
                this.status = VolumeOperationStatus.SUCCEEDED;
            } else {
                LOGGER.error("Failed to upload file {%s} with error code: {%s}", this.localFilePath, response.getStatusLine().getStatusCode());
                this.status = VolumeOperationStatus.FAILED;
                this.errorMessage = "Failed to upload file with error code: " + response.getStatusLine().getStatusCode();
            }
        }
        catch (DatabricksHttpException | IOException e) {
            LOGGER.error("Failed to upload file {} with error {}", this.localFilePath, e.getMessage());
            this.status = VolumeOperationStatus.FAILED;
            this.errorMessage = "Failed to upload file: " + e.getMessage();
        }
    }

    private boolean localFileHasErrorForPutOperation(File file) {
        if (!file.exists() || file.isDirectory()) {
            LOGGER.error("Local file does not exist or is a directory {}", this.localFilePath);
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Local file does not exist or is a directory";
            return true;
        }
        if (file.length() == 0L) {
            LOGGER.error("Local file is empty {}", this.localFilePath);
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Local file is empty";
            return true;
        }
        if (file.length() > PUT_SIZE_LIMITS) {
            LOGGER.error("Local file too large {}", this.localFilePath);
            this.status = VolumeOperationStatus.ABORTED;
            this.errorMessage = "Local file too large";
            return true;
        }
        return false;
    }

    private void executeDeleteOperation() {
        HttpDelete httpDelete = new HttpDelete(this.operationUrl);
        this.headers.forEach(httpDelete::addHeader);
        try (CloseableHttpResponse response = this.databricksHttpClient.execute(httpDelete);){
            if (HttpUtil.isSuccessfulHttpResponse(response)) {
                this.status = VolumeOperationStatus.SUCCEEDED;
            } else {
                LOGGER.error("Failed to delete volume with error code: {%s}", response.getStatusLine().getStatusCode());
                this.status = VolumeOperationStatus.FAILED;
                this.errorMessage = "Failed to delete volume";
            }
        }
        catch (DatabricksHttpException | IOException e) {
            LOGGER.error(e, "Failed to delete volume with error {}", e.getMessage());
            this.status = VolumeOperationStatus.FAILED;
            this.errorMessage = "Failed to delete volume: " + e.getMessage();
        }
    }

    public static class Builder {
        private VolumeUtil.VolumeOperationType operationType;
        private String operationUrl;
        private String localFilePath = null;
        private Map<String, String> headers = new HashMap<String, String>();
        private Set<String> allowedVolumeIngestionPaths = null;
        private boolean enableVolumeOperations = false;
        private boolean isAllowedInputStreamForVolumeOperation = false;
        private IDatabricksHttpClient databricksHttpClient = null;
        private InputStreamEntity inputStream = null;
        private Consumer<HttpEntity> getStreamReceiver = null;
        private VolumeOperationStatus status = VolumeOperationStatus.PENDING;
        private String errorMessage = null;

        public static Builder createBuilder() {
            return new Builder();
        }

        public Builder operationType(VolumeUtil.VolumeOperationType operationType) {
            this.operationType = operationType;
            return this;
        }

        public Builder operationUrl(String operationUrl) {
            this.operationUrl = operationUrl;
            return this;
        }

        public Builder localFilePath(String localFilePath) {
            this.localFilePath = localFilePath;
            return this;
        }

        public Builder headers(Map<String, String> headers) {
            this.headers = headers;
            return this;
        }

        public Builder allowedVolumeIngestionPathString(String allowedVolumeIngestionPathString) {
            this.allowedVolumeIngestionPaths = VolumeOperationProcessor.getAllowedPaths(allowedVolumeIngestionPathString);
            return this;
        }

        public Builder isAllowedInputStreamForVolumeOperation(boolean isAllowedInputStreamForVolumeOperation) {
            this.isAllowedInputStreamForVolumeOperation = isAllowedInputStreamForVolumeOperation;
            return this;
        }

        public Builder isEnableVolumeOperations(boolean enableVolumeOperations) {
            this.enableVolumeOperations = enableVolumeOperations;
            return this;
        }

        public Builder databricksHttpClient(IDatabricksHttpClient databricksHttpClient) {
            this.databricksHttpClient = databricksHttpClient;
            return this;
        }

        public Builder inputStream(InputStreamEntity inputStream) {
            this.inputStream = inputStream;
            return this;
        }

        public Builder getStreamReceiver(Consumer<HttpEntity> getStreamReceiver) {
            this.getStreamReceiver = getStreamReceiver;
            return this;
        }

        public Builder status(VolumeOperationStatus status) {
            this.status = status;
            return this;
        }

        public Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public VolumeOperationProcessor build() {
            return new VolumeOperationProcessor(this);
        }
    }
}

