package com.codeborne.selenide.impl;

import com.codeborne.selenide.CheckResult;
import com.codeborne.selenide.Condition;
import com.codeborne.selenide.Driver;
import com.codeborne.selenide.SelenideElement;
import com.codeborne.selenide.ex.ElementNotFound;
import com.codeborne.selenide.ex.ElementShould;
import com.codeborne.selenide.ex.ElementShouldNot;
import com.codeborne.selenide.ex.UIAssertionError;
import org.openqa.selenium.By;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.WebElement;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.List;

import static com.codeborne.selenide.CheckResult.Verdict.ACCEPT;
import static com.codeborne.selenide.Condition.editable;
import static com.codeborne.selenide.Condition.interactable;
import static com.codeborne.selenide.Condition.not;
import static com.codeborne.selenide.impl.Alias.NONE;
import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNull;

@ParametersAreNonnullByDefault
public abstract class WebElementSource {
  @Nonnull
  private Alias alias = NONE;

  @CheckReturnValue
  @Nonnull
  public abstract Driver driver();

  @CheckReturnValue
  @Nonnull
  public abstract WebElement getWebElement();

  @CheckReturnValue
  @Nonnull
  public abstract String getSearchCriteria();

  public void setAlias(String alias) {
    this.alias = new Alias(alias);
  }

  @CheckReturnValue
  @Nonnull
  public Alias getAlias() {
    return alias;
  }

  @CheckReturnValue
  @Nonnull
  public String description() {
    return alias.getOrElse(this::getSearchCriteria);
  }

  @Override
  @CheckReturnValue
  @Nonnull
  public String toString() {
    return description();
  }

  @CheckReturnValue
  @Nonnull
  public SelenideElement find(SelenideElement proxy, Object arg, int index) {
    return ElementFinder.wrap(driver(), this, getSelector(arg), index);
  }

  @CheckReturnValue
  @Nonnull
  public List<WebElement> findAll() throws IndexOutOfBoundsException {
    return singletonList(getWebElement());
  }

  @CheckReturnValue
  @Nonnull
  public ElementNotFound createElementNotFoundError(Condition condition, Throwable lastError) {
    if (lastError instanceof UIAssertionError) {
      throw new IllegalArgumentException("Unexpected UIAssertionError as a cause of ElementNotFound: " + lastError, lastError);
    }
    return new ElementNotFound(alias, getSearchCriteria(), condition, lastError);
  }

  @CheckReturnValue
  @Nonnull
  public static By getSelector(Object arg) {
    return arg instanceof By ? (By) arg : By.cssSelector((String) arg);
  }

  @Nullable
  @CheckReturnValue
  public WebElement checkCondition(String prefix, Condition condition, boolean invert) {
    Condition check = invert ? not(condition) : condition;

    Throwable lastError = null;
    WebElement element = null;
    CheckResult checkResult = null;
    try {
      element = getWebElement();
      checkResult = check.check(driver(), element);

      if (checkResult.verdict() == ACCEPT) {
        return element;
      }
    }
    catch (WebDriverException | IndexOutOfBoundsException | AssertionError e) {
      lastError = e;
    }

    if (lastError != null && Cleanup.of.isInvalidSelectorError(lastError)) {
      throw Cleanup.of.wrapInvalidSelectorException(lastError);
    }

    if (element == null) {
      if (check.missingElementSatisfiesCondition()) {
        return null;
      }
      throw createElementNotFoundError(check, lastError);
    }
    else if (invert) {
      throw new ElementShouldNot(driver(), description(), prefix, condition, checkResult, element, lastError);
    }
    else {
      throw new ElementShould(driver(), description(), prefix, condition, checkResult, element, lastError);
    }
  }

  /**
   * Asserts that returned element can be interacted with.
   *
   * @return element or throws ElementShould/ElementShouldNot exceptions
   */
  @Nonnull
  @CheckReturnValue
  public WebElement findAndAssertElementIsInteractable() {
    return requireNonNull(checkCondition("be ", interactable, false));
  }

  /**
   * Asserts that returned element is editable.
   * @return element or throws ElementShould/ElementShouldNot exceptions
   * @since 6.5.0
   */
  @Nonnull
  @CheckReturnValue
  public WebElement findAndAssertElementIsEditable() {
    return requireNonNull(checkCondition("be ", editable, false));
  }
}
