// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.security.attestation.models;

import com.azure.core.annotation.Immutable;
import com.azure.core.util.BinaryData;

import java.time.OffsetDateTime;

/**
 * An AttestationResult contains the defined claims included in a Microsoft Azure Attestation Token.
 *
 * An AttestationResult is an IETF RFC 7519 JSON Web Token, and implements many of the claims in
 * IETF Draft-ietf-rats-eat.
 */
@Immutable
public interface AttestationResult {
    /**
     * Get the Unique Identifier property: The Unique Identifier corresponds to the "jti" claim in
     * a Json Web Token.
     *
     * @return the jti value. See <a href="https://datatracker.ietf.org/doc/html/rfc7519#section-4.1.7">RFC 7519 section 4.1.7</a>
     * for more information.
     */
    String getUniqueIdentifier();

    /**
     * Retrieve the issuer of the attestation token. The issuer corresponds to the "iss" claim
     * in a Json Web Token. See <a href="https://datatracker.ietf.org/doc/html/rfc7519#section-4.1.1">RFC 7519 section 4.1.1</a>
     * for more information.
     *
     * The issuer will always be the same as the attestation service instance endpoint URL.
     *
     * @return the iss value.
     */
    String getIssuer();

    /**
     * Get the Issued At property: The time at which the token was issued. The IssuedAt property
     * corresponds to the "iat" claim in a Json Web Token. See <a href="https://datatracker.ietf.org/doc/html/rfc7519#section-4.1.6">RFC 7519 section 4.1.6</a>
     * for more information.
     *
     * @return the IssuedAt value.
     */
    OffsetDateTime getIssuedAt();

    /**
     * Get the ExpiresOn property: The expiration time after which the token is no longer valid. The ExpiresOn property
     * corresponds to the "exp" claim in a Json Web Token.  See <a href="https://datatracker.ietf.org/doc/html/rfc7519#section-4.1.4">RFC 7519 section 4.1.4</a>
     *
     * @return the expiration time for the token.
     */
    OffsetDateTime getExpiresOn();

    /**
     * Get the NotBefore property: The time before which a token cannot be considered valid. The ExpiresOn property
     * corresponds to the "exp" claim in a Json Web Token.  See <a href="https://datatracker.ietf.org/doc/html/rfc7519#section-4.1.4">RFC 7519 section 4.1.4</a>
     *
     * @return the time before which the token is invalid.
     */
    OffsetDateTime getNotBefore();

    /**
     * Returns the  Nonce input to the attestation request, if provided. The Nonce claim corresponds to the
     * "nonce" claim defined in <a href="https://datatracker.ietf.org/doc/html/draft-ietf-rats-eat-10#section-3.3">draft-ietf-rats-eat-10, section 3.3</a>
     *
     * @return the nonce value.
     */
    String getNonce();

    /**
     * Get the version property: The Schema version of this structure. Current Value: 1.0.
     *
     * @return the version value.
     */
    String getVersion();

    /**
     * Get the runtimeClaims property: Runtime Claims.
     *
     * If {@link AttestationDataInterpretation#JSON} was specified in the {@link AttestationData} for the RunTime data
     * this will contain the input RunTimeData as JSON elements.
     *
     * @return the RunTimeClaims value.
     */
    Object getRuntimeClaims();

    /**
     * Get the initTimeClaims property: InitTime Claims.
     * <br>
     * If {@link AttestationDataInterpretation#JSON} was specified in the {@link AttestationData} for the InitTime data
     * this will contain the input InitTimeData as JSON elements.
     *
     * @return the initTimeClaims value.
     */
    Object getInitTimeClaims();

    /**
     * Get the policyClaims property: Policy Generated Claims. This element contains all claims
     * in the token which were generated by customer specified attestation policies.
     *
     * @return the policyClaims value.
     */
    Object getPolicyClaims();

    /**
     * Get the verifierType property: The Attestation Type being attested. Normally this will be
     * one of "sgx", or "tpm"
     *
     * @return the verifierType value.
     */
    String getVerifierType();

    /**
     * Get the policySigner property: If the attestation policy was signed, this will
     * contain the certificate used to sign the policy object.
     *
     * @return the policySigner value.
     */
    AttestationSigner getPolicySigner();

    /**
     * Get the policyHash property: The SHA256 hash of the BASE64URL encoded policy text used for attestation.
     *
     * @return the policyHash value.
     */
    BinaryData getPolicyHash();

    /**
     * Get the isDebuggable property: True if the enclave can be debugged, false otherwise.
     *
     * If the attestation evidence came from OpenEnclave, this corresponds to the OE_REPORT_ATTRIBUTES_DEBUG flag
     * in the generated OpenEnclave report (<a href="https://openenclave.io/apidocs/v0.17/struct__oe__identity_aaf1cd8db46a5f4f4156c97bfe265d3cd.html">oe_identity::attributes</a>
     *
     * If the attestation evidence came from the Intel SGX SDK, this corresponds to the SGX_FLAGS_DEBUG flag.
     * @return the isDebuggable value.
     */
    Boolean isDebuggable();

    /**
     * Get the productId property: The SGX Product ID for the enclave.
     *
     * The Product ID represents the ISVPRODID value from the enclave. See <a href="https://openenclave.github.io/openenclave/api/struct__oe__identity_a7e04eefe0f0302d2f8c71dc3d06fd476.html">oe_identity::product_id</a>
     * for more information.
     *
     * @return the productId value.
     */
    int getProductId();

    /**
     * Get the mrEnclave property: The HEX encoded SGX MRENCLAVE value for the enclave.
     *
     * Represents the HEX encoded MRENCLAVE Measure Register. This is the hash of the binary loaded
     * into the enclave.
     *
     * @return the mrEnclave value.
     */
    String getMrEnclave();

    /**
     * Get the mrSigner property: The HEX encoded SGX MRSIGNER value for the enclave.

     * Represents the HEX encoded MRSIGNER Measure Register. This is the hash of the signer of the
     * code loaded into the enclave.
     *
     * @return the mrSigner value.
     */
    String getMrSigner();

    /**
     * Get the svn property: The SGX SVN value for the enclave.
     *
     * The SVN for the enclave should be incremented on every security update.
     *
     * @return the svn value.
     */
    int getSvn();

    /**
     * Get the enclaveHeldData property: A copy of the RuntimeData specified as an input to the call to attest if
     * the {@link AttestationDataInterpretation#BINARY} data interpretation was set on the {@link AttestationData} passed to
     * the {@link AttestationOptions#setRunTimeData(AttestationData)} API.
     *
     * @return the enclaveHeldData value.
     */
    BinaryData getEnclaveHeldData();

    /**
     * Get the sgxCollateral property: The SGX SVN value for the enclave.
     *
     * The SGX Collateral contains the HEX encoded SHA256 hash of each of the inputs to the attestation token.
     *
     * @return the sgxCollateral value.
     */
    Object getSgxCollateral();

}
