// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.core.amqp.implementation;

import com.azure.core.amqp.AmqpShutdownSignal;
import com.azure.core.util.logging.ClientLogger;

import static com.azure.core.amqp.implementation.AmqpLoggingUtils.addShutdownSignal;

/**
 * Handles exceptions generated by AMQP connections, sessions, and/or links.
 */
abstract class AmqpExceptionHandler {
    private static final ClientLogger LOGGER = new ClientLogger(AmqpExceptionHandler.class);

    /**
     * Creates a new instance of the exception handler.
     */
    AmqpExceptionHandler() {
    }

    /**
     * Notifies the exception handler of an exception.
     *
     * @param exception The exception that caused the connection error.
     */
    void onConnectionError(Throwable exception) {
        LOGGER.warning("Connection exception encountered.", exception);
    }

    /**
     * Notifies the exception handler that a shutdown signal occurred.
     *
     * @param shutdownSignal The shutdown signal that was received.
     */
    void onConnectionShutdown(AmqpShutdownSignal shutdownSignal) {
        addShutdownSignal(LOGGER.atInfo(), shutdownSignal).log("Shutdown received");
    }
}
