/*
 * Decompiled with CFR 0.152.
 */
package com.akamai.edgegrid.signer;

import com.akamai.edgegrid.signer.ClientCredential;
import com.akamai.edgegrid.signer.Request;
import com.akamai.edgegrid.signer.exceptions.RequestSigningException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.Base64;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EdgeGridV1Signer {
    private static final String ALGORITHM_NAME = "EG1-HMAC-SHA256";
    private static final Pattern PATTERN_SPACES = Pattern.compile("\\s+");
    private static final String AUTH_CLIENT_TOKEN_NAME = "client_token";
    private static final String AUTH_ACCESS_TOKEN_NAME = "access_token";
    private static final String AUTH_TIMESTAMP_NAME = "timestamp";
    private static final String AUTH_NONCE_NAME = "nonce";
    private static final String AUTH_SIGNATURE_NAME = "signature";
    private static final String DIGEST_ALGORITHM = "SHA-256";
    private static final String SIGNING_ALGORITHM = "HmacSHA256";
    private static final Logger log = LoggerFactory.getLogger(EdgeGridV1Signer.class);

    public String getSignature(Request request, ClientCredential credential) throws RequestSigningException {
        return this.getSignature(request, credential, System.currentTimeMillis(), EdgeGridV1Signer.generateNonce());
    }

    private static String generateNonce() {
        return UUID.randomUUID().toString();
    }

    private static String getAuthorizationHeaderValue(String authData, String signature) {
        return authData + AUTH_SIGNATURE_NAME + '=' + signature;
    }

    private static String getRelativePathWithQuery(URI uri) {
        StringBuilder sb = new StringBuilder(uri.getRawPath());
        if (uri.getQuery() != null) {
            sb.append("?").append(uri.getQuery());
        }
        return sb.toString();
    }

    private static byte[] sign(String s, String clientSecret) throws RequestSigningException {
        return EdgeGridV1Signer.sign(s, clientSecret.getBytes(StandardCharsets.UTF_8));
    }

    private static byte[] sign(String s, byte[] key) throws RequestSigningException {
        try {
            SecretKeySpec signingKey = new SecretKeySpec(key, SIGNING_ALGORITHM);
            Mac mac = Mac.getInstance(SIGNING_ALGORITHM);
            mac.init(signingKey);
            byte[] valueBytes = s.getBytes(StandardCharsets.UTF_8);
            return mac.doFinal(valueBytes);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RequestSigningException("Failed to sign: your JDK does not recognize signing algorithm <HmacSHA256>", e);
        }
        catch (InvalidKeyException e) {
            throw new RequestSigningException("Failed to sign: invalid key", e);
        }
    }

    private static String formatTimeStamp(long time) {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd'T'HH:mm:ssZ");
        Date date = new Date(time);
        format.setTimeZone(TimeZone.getTimeZone("UTC"));
        return format.format(date);
    }

    private static String canonicalizeUri(String uri) {
        if (uri == null || "".equals(uri)) {
            return "/";
        }
        if (uri.charAt(0) != '/') {
            uri = "/" + uri;
        }
        return uri;
    }

    String getSignature(Request request, ClientCredential credential, long timestamp, String nonce) throws RequestSigningException {
        Objects.requireNonNull(credential, "credential cannot be null");
        Objects.requireNonNull(request, "request cannot be null");
        String timeStamp = EdgeGridV1Signer.formatTimeStamp(timestamp);
        String authData = this.getAuthData(credential, timeStamp, nonce);
        String signature = this.getSignature(request, credential, timeStamp, authData);
        log.debug("Signature: {}", (Object)signature);
        return EdgeGridV1Signer.getAuthorizationHeaderValue(authData, signature);
    }

    private String getSignature(Request request, ClientCredential credential, String timeStamp, String authData) throws RequestSigningException {
        String signingKey = this.getSigningKey(timeStamp, credential.getClientSecret());
        String canonicalizedRequest = this.getCanonicalizedRequest(request, credential);
        log.trace("Canonicalized request: {}", (Object)canonicalizedRequest);
        String dataToSign = this.getDataToSign(canonicalizedRequest, authData);
        log.trace("Data to sign: {}", (Object)dataToSign);
        return this.signAndEncode(dataToSign, signingKey);
    }

    private String signAndEncode(String stringToSign, String signingKey) throws RequestSigningException {
        byte[] signatureBytes = EdgeGridV1Signer.sign(stringToSign, signingKey);
        return Base64.getEncoder().encodeToString(signatureBytes);
    }

    private String getSigningKey(String timeStamp, String clientSecret) throws RequestSigningException {
        byte[] signingKeyBytes = EdgeGridV1Signer.sign(timeStamp, clientSecret);
        return Base64.getEncoder().encodeToString(signingKeyBytes);
    }

    private String getDataToSign(String canonicalizedRequest, String authData) {
        return canonicalizedRequest + authData;
    }

    private String getAuthData(ClientCredential credential, String timeStamp, String nonce) {
        StringBuilder sb = new StringBuilder();
        sb.append(ALGORITHM_NAME);
        sb.append(' ');
        sb.append(AUTH_CLIENT_TOKEN_NAME);
        sb.append('=');
        sb.append(credential.getClientToken());
        sb.append(';');
        sb.append(AUTH_ACCESS_TOKEN_NAME);
        sb.append('=');
        sb.append(credential.getAccessToken());
        sb.append(';');
        sb.append(AUTH_TIMESTAMP_NAME);
        sb.append('=');
        sb.append(timeStamp);
        sb.append(';');
        sb.append(AUTH_NONCE_NAME);
        sb.append('=');
        sb.append(nonce);
        sb.append(';');
        return sb.toString();
    }

    private String getCanonicalizedRequest(Request request, ClientCredential credential) throws RequestSigningException {
        StringBuilder sb = new StringBuilder();
        sb.append(request.getMethod().toUpperCase());
        sb.append('\t');
        String scheme = "https";
        sb.append(scheme);
        sb.append('\t');
        String host = credential.getHost();
        sb.append(host.toLowerCase());
        sb.append('\t');
        String relativePath = EdgeGridV1Signer.getRelativePathWithQuery(request.getUri());
        String relativeUrl = EdgeGridV1Signer.canonicalizeUri(relativePath);
        sb.append(relativeUrl);
        sb.append('\t');
        String canonicalizedHeaders = this.canonicalizeHeaders(request.getHeaders(), credential);
        sb.append(canonicalizedHeaders);
        sb.append('\t');
        sb.append(this.getContentHash(request.getMethod(), request.getBody(), credential.getMaxBodySize()));
        sb.append('\t');
        return sb.toString();
    }

    private byte[] getHash(byte[] requestBody, int offset, int len) throws RequestSigningException {
        try {
            MessageDigest md = MessageDigest.getInstance(DIGEST_ALGORITHM);
            md.update(requestBody, offset, len);
            return md.digest();
        }
        catch (NoSuchAlgorithmException e) {
            throw new RequestSigningException("Failed to get request hash: your JDK does not recognize algorithm <SHA-256>", e);
        }
    }

    private String canonicalizeHeaders(Map<String, String> requestHeaders, ClientCredential credential) {
        return requestHeaders.entrySet().stream().filter(entry -> credential.getHeadersToSign().contains(entry.getKey())).filter(entry -> entry.getValue() != null).filter(entry -> !"".equals(entry.getValue())).map(entry -> ((String)entry.getKey()).toLowerCase() + ":" + this.canonicalizeHeaderValue((String)entry.getValue())).collect(Collectors.joining("\t"));
    }

    private String canonicalizeHeaderValue(String headerValue) {
        if ((headerValue = headerValue.trim()) != null && !"".equals(headerValue)) {
            Matcher matcher = PATTERN_SPACES.matcher(headerValue);
            headerValue = matcher.replaceAll(" ");
        }
        return headerValue;
    }

    private String getContentHash(String requestMethod, byte[] requestBody, int maxBodySize) throws RequestSigningException {
        if (!"POST".equals(requestMethod)) {
            return "";
        }
        if (requestBody == null || requestBody.length == 0) {
            return "";
        }
        int lengthToHash = requestBody.length;
        if (lengthToHash > maxBodySize) {
            log.info("Content length '{}' exceeds signing length of '{}'. Less than the entire message will be signed.", (Object)lengthToHash, (Object)maxBodySize);
            lengthToHash = maxBodySize;
        } else if (log.isTraceEnabled()) {
            log.trace("Content (Base64): {}", (Object)Base64.getEncoder().encodeToString(requestBody));
        }
        byte[] digestBytes = this.getHash(requestBody, 0, lengthToHash);
        log.debug("Content hash (Base64): {}", (Object)Base64.getEncoder().encodeToString(digestBytes));
        return Base64.getEncoder().encodeToString(digestBytes);
    }
}

