/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.spi.security.user.util;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.util.Objects;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.apache.jackrabbit.guava.common.base.Strings;
import org.apache.jackrabbit.oak.spi.security.ConfigurationParameters;
import org.apache.jackrabbit.util.Text;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PasswordUtil {
    private static final Logger log = LoggerFactory.getLogger(PasswordUtil.class);
    private static final char DELIMITER = '-';
    private static final int NO_ITERATIONS = 1;
    private static final String ENCODING = "UTF-8";
    private static final int PBKDF2_KEY_LENGTH = 128;
    static final String PBKDF2_PREFIX = "PBKDF2";
    public static final String DEFAULT_ALGORITHM = "SHA-256";
    public static final int DEFAULT_SALT_SIZE = 8;
    public static final int DEFAULT_ITERATIONS = 1000;

    private PasswordUtil() {
    }

    public static String buildPasswordHash(@NotNull String password) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        return PasswordUtil.buildPasswordHash(password, DEFAULT_ALGORITHM, 8, 1000);
    }

    public static String buildPasswordHash(@NotNull String password, @Nullable String algorithm, int saltSize, int iterations) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        Objects.requireNonNull(password);
        if (iterations < 1) {
            iterations = 1000;
        }
        if (saltSize < 8) {
            saltSize = 8;
        }
        String salt = PasswordUtil.generateSalt(saltSize);
        String alg = algorithm == null ? DEFAULT_ALGORITHM : algorithm;
        return PasswordUtil.generateHash(password, alg, salt, iterations);
    }

    public static String buildPasswordHash(@NotNull String password, @NotNull ConfigurationParameters config) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        Objects.requireNonNull(config);
        String algorithm = config.getConfigValue("passwordHashAlgorithm", DEFAULT_ALGORITHM);
        int iterations = config.getConfigValue("passwordHashIterations", 1000);
        int saltSize = config.getConfigValue("passwordSaltSize", 8);
        return PasswordUtil.buildPasswordHash(password, algorithm, saltSize, iterations);
    }

    public static boolean isPlainTextPassword(@Nullable String password) {
        return PasswordUtil.extractAlgorithm(password) == null;
    }

    public static boolean isSame(@Nullable String hashedPassword, @NotNull char[] password) {
        return PasswordUtil.isSame(hashedPassword, String.valueOf(password));
    }

    public static boolean isSame(@Nullable String hashedPassword, @NotNull String password) {
        if (hashedPassword == null) {
            return false;
        }
        try {
            String algorithm = PasswordUtil.extractAlgorithm(hashedPassword);
            if (algorithm != null) {
                int startPos = algorithm.length() + 2;
                String salt = PasswordUtil.extractSalt(hashedPassword, startPos);
                int iterations = 1;
                if (salt != null) {
                    iterations = PasswordUtil.extractIterations(hashedPassword, startPos += salt.length() + 1);
                }
                String hash = PasswordUtil.generateHash(password, algorithm, salt, iterations);
                return PasswordUtil.compareSecure(hashedPassword, hash);
            }
        }
        catch (UnsupportedEncodingException | NoSuchAlgorithmException e) {
            log.warn(e.getMessage());
        }
        return false;
    }

    private static boolean compareSecure(@NotNull String a, @NotNull String b) {
        int len = a.length();
        if (len != b.length()) {
            return false;
        }
        if (len == 0) {
            return true;
        }
        int bits = 0;
        for (int i = 0; i < len; ++i) {
            bits |= a.charAt(i) ^ b.charAt(i);
        }
        return bits == 0;
    }

    @NotNull
    private static String generateHash(@NotNull String pwd, @NotNull String algorithm, @Nullable String salt, int iterations) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        StringBuilder passwordHash = new StringBuilder();
        passwordHash.append('{').append(algorithm).append('}');
        if (!Strings.isNullOrEmpty((String)salt)) {
            StringBuilder data = new StringBuilder();
            data.append(salt).append(pwd);
            passwordHash.append(salt).append('-');
            if (iterations > 1) {
                passwordHash.append(iterations).append('-');
            }
            String digest = algorithm.startsWith(PBKDF2_PREFIX) ? PasswordUtil.generatePBKDF2(pwd, salt, algorithm, iterations) : PasswordUtil.generateDigest(data.toString(), algorithm, iterations);
            passwordHash.append(digest);
        } else {
            passwordHash.append(Text.digest(algorithm, pwd.getBytes(ENCODING)));
        }
        return passwordHash.toString();
    }

    @NotNull
    private static String generateSalt(int saltSize) {
        SecureRandom random = new SecureRandom();
        byte[] salt = new byte[saltSize];
        random.nextBytes(salt);
        return PasswordUtil.convertBytesToHex(salt);
    }

    @NotNull
    private static String convertBytesToHex(byte[] bytes) {
        StringBuilder res = new StringBuilder(bytes.length * 2);
        for (byte b : bytes) {
            res.append(Text.hexTable[b >> 4 & 0xF]);
            res.append(Text.hexTable[b & 0xF]);
        }
        return res.toString();
    }

    @NotNull
    private static byte[] convertHexToBytes(String s) {
        int len = s.length();
        if (len % 2 != 0) {
            throw new IllegalArgumentException("Not a hex encoded byte array: " + s);
        }
        byte[] bytes = new byte[len / 2];
        for (int i = 0; i < bytes.length; ++i) {
            bytes[i] = (byte)((Character.digit(s.charAt(i + i), 16) << 4) + Character.digit(s.charAt(i + i + 1), 16));
        }
        return bytes;
    }

    @NotNull
    private static String generatePBKDF2(@NotNull String pwd, @NotNull String salt, @NotNull String algorithm, int iterations) throws NoSuchAlgorithmException {
        SecretKeyFactory factory = SecretKeyFactory.getInstance(algorithm);
        byte[] saltBytes = PasswordUtil.convertHexToBytes(salt);
        PBEKeySpec keyspec = new PBEKeySpec(pwd.toCharArray(), saltBytes, iterations, 128);
        try {
            SecretKey key = factory.generateSecret(keyspec);
            byte[] bytes = key.getEncoded();
            return PasswordUtil.convertBytesToHex(bytes);
        }
        catch (InvalidKeySpecException e) {
            throw new NoSuchAlgorithmException(algorithm, e);
        }
    }

    @NotNull
    private static String generateDigest(@NotNull String data, @NotNull String algorithm, int iterations) throws UnsupportedEncodingException, NoSuchAlgorithmException {
        byte[] bytes = data.getBytes(ENCODING);
        MessageDigest md = MessageDigest.getInstance(algorithm);
        for (int i = 0; i < iterations; ++i) {
            md.reset();
            bytes = md.digest(bytes);
        }
        return PasswordUtil.convertBytesToHex(bytes);
    }

    @Nullable
    private static String extractAlgorithm(@Nullable String hashedPwd) {
        if (!Strings.isNullOrEmpty((String)hashedPwd)) {
            int end = hashedPwd.indexOf(125);
            if (hashedPwd.charAt(0) == '{' && end > 0 && end < hashedPwd.length() - 1) {
                String algorithm = hashedPwd.substring(1, end);
                try {
                    if (algorithm.startsWith(PBKDF2_PREFIX)) {
                        SecretKeyFactory.getInstance(algorithm);
                    } else {
                        MessageDigest.getInstance(algorithm);
                    }
                    return algorithm;
                }
                catch (NoSuchAlgorithmException e) {
                    log.debug("Invalid algorithm detected {}", (Object)algorithm, (Object)e);
                }
            }
        }
        return null;
    }

    @Nullable
    private static String extractSalt(@NotNull String hashedPwd, int start) {
        int end = hashedPwd.indexOf(45, start);
        if (end > -1) {
            return hashedPwd.substring(start, end);
        }
        return null;
    }

    private static int extractIterations(@NotNull String hashedPwd, int start) {
        int end = hashedPwd.indexOf(45, start);
        if (end > -1) {
            String str = hashedPwd.substring(start, end);
            try {
                return Integer.parseInt(str);
            }
            catch (NumberFormatException e) {
                log.debug("Expected number of iterations. Found: {}", (Object)str, (Object)e);
            }
        }
        return 1;
    }
}

