/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.tag;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.ResolverStyle;
import java.time.format.SignStyle;
import java.time.temporal.ChronoField;
import java.util.Optional;
import java.util.SortedMap;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.Snapshot;
import org.apache.paimon.data.Timestamp;
import org.apache.paimon.operation.TagDeletion;
import org.apache.paimon.shade.guava30.com.google.common.base.MoreObjects;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SnapshotManager;
import org.apache.paimon.utils.TagManager;

public class TagAutoCreation {
    private static final DateTimeFormatter HOUR_FORMATTER = new DateTimeFormatterBuilder().appendValue(ChronoField.YEAR, 1, 10, SignStyle.NORMAL).appendLiteral('-').appendValue(ChronoField.MONTH_OF_YEAR, 2, 2, SignStyle.NORMAL).appendLiteral('-').appendValue(ChronoField.DAY_OF_MONTH, 2, 2, SignStyle.NORMAL).appendLiteral(" ").appendValue(ChronoField.HOUR_OF_DAY, 2, 2, SignStyle.NORMAL).toFormatter().withResolverStyle(ResolverStyle.LENIENT);
    private static final DateTimeFormatter DAY_FORMATTER = new DateTimeFormatterBuilder().appendValue(ChronoField.YEAR, 1, 10, SignStyle.NORMAL).appendLiteral('-').appendValue(ChronoField.MONTH_OF_YEAR, 2, 2, SignStyle.NORMAL).appendLiteral('-').appendValue(ChronoField.DAY_OF_MONTH, 2, 2, SignStyle.NORMAL).toFormatter().withResolverStyle(ResolverStyle.LENIENT);
    private final SnapshotManager snapshotManager;
    private final TagManager tagManager;
    private final TagDeletion tagDeletion;
    private final TimeExtractor timeExtractor;
    private final TagPeriodHandler periodHandler;
    private final Duration delay;
    private final Integer numRetainedMax;
    private LocalDateTime nextTag;
    private long nextSnapshot;

    private TagAutoCreation(SnapshotManager snapshotManager, TagManager tagManager, TagDeletion tagDeletion, TimeExtractor timeExtractor, TagPeriodHandler periodHandler, Duration delay, Integer numRetainedMax) {
        this.snapshotManager = snapshotManager;
        this.tagManager = tagManager;
        this.tagDeletion = tagDeletion;
        this.timeExtractor = timeExtractor;
        this.periodHandler = periodHandler;
        this.delay = delay;
        this.numRetainedMax = numRetainedMax;
        this.periodHandler.validateDelay(delay);
        SortedMap<Snapshot, String> tags = tagManager.tags();
        if (tags.isEmpty()) {
            this.nextSnapshot = MoreObjects.firstNonNull(snapshotManager.earliestSnapshotId(), 1L);
        } else {
            Snapshot lastTag = tags.lastKey();
            this.nextSnapshot = lastTag.id() + 1L;
            LocalDateTime time = periodHandler.tagToTime((String)tags.get(lastTag));
            this.nextTag = periodHandler.nextTagTime(time);
        }
    }

    public void run() {
        while (true) {
            if (this.snapshotManager.snapshotExists(this.nextSnapshot)) {
                this.tryToTag(this.snapshotManager.snapshot(this.nextSnapshot));
                ++this.nextSnapshot;
                continue;
            }
            Long earliest = this.snapshotManager.earliestSnapshotId();
            if (earliest == null || earliest <= this.nextSnapshot) break;
            this.nextSnapshot = earliest;
        }
    }

    private void tryToTag(Snapshot snapshot) {
        Optional<LocalDateTime> timeOptional = this.timeExtractor.extract(snapshot);
        if (!timeOptional.isPresent()) {
            return;
        }
        LocalDateTime time = timeOptional.get();
        if (this.nextTag == null || this.isAfterOrEqual(time.minus(this.delay), this.periodHandler.nextTagTime(this.nextTag))) {
            SortedMap<Snapshot, String> tags;
            LocalDateTime thisTag = this.periodHandler.normalizeToTagTime(time);
            String tagName = this.periodHandler.timeToTag(thisTag);
            this.tagManager.createTag(snapshot, tagName);
            this.nextTag = this.periodHandler.nextTagTime(thisTag);
            if (this.numRetainedMax != null && (tags = this.tagManager.tags()).size() > this.numRetainedMax) {
                int toDelete = tags.size() - this.numRetainedMax;
                int i = 0;
                for (String tag : tags.values()) {
                    this.tagManager.deleteTag(tag, this.tagDeletion, this.snapshotManager);
                    if (++i != toDelete) continue;
                    break;
                }
            }
        }
    }

    private boolean isAfterOrEqual(LocalDateTime t1, LocalDateTime t2) {
        return t1.isAfter(t2) || t1.isEqual(t2);
    }

    @Nullable
    public static TagAutoCreation create(CoreOptions options, SnapshotManager snapshotManager, TagManager tagManager, TagDeletion tagDeletion) {
        BaseTagPeriodHandler periodHandler;
        TimeExtractor timeExtractor;
        switch (options.tagCreationMode()) {
            case NONE: {
                return null;
            }
            case PROCESS_TIME: {
                timeExtractor = new ProcessTimeExtractor();
                break;
            }
            case WATERMARK: {
                timeExtractor = new WatermarkExtractor(ZoneId.of(options.sinkWatermarkTimeZone()));
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unsupported " + options.tagCreationMode());
            }
        }
        switch (options.tagCreationPeriod()) {
            case DAILY: {
                periodHandler = new DailyTagPeriodHandler();
                break;
            }
            case HOURLY: {
                periodHandler = new HourlyTagPeriodHandler();
                break;
            }
            case TWO_HOURS: {
                periodHandler = new TwoHoursTagPeriodHandler();
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unsupported " + options.tagCreationPeriod());
            }
        }
        return new TagAutoCreation(snapshotManager, tagManager, tagDeletion, timeExtractor, periodHandler, options.tagCreationDelay(), options.tagNumRetainedMax());
    }

    private static class TwoHoursTagPeriodHandler
    extends BaseTagPeriodHandler {
        private static final Duration ONE_PERIOD = Duration.ofHours(2L);

        private TwoHoursTagPeriodHandler() {
        }

        @Override
        protected Duration onePeriod() {
            return ONE_PERIOD;
        }

        @Override
        protected DateTimeFormatter formatter() {
            return HOUR_FORMATTER;
        }
    }

    private static class DailyTagPeriodHandler
    extends BaseTagPeriodHandler {
        private static final Duration ONE_PERIOD = Duration.ofDays(1L);

        private DailyTagPeriodHandler() {
        }

        @Override
        protected Duration onePeriod() {
            return ONE_PERIOD;
        }

        @Override
        protected DateTimeFormatter formatter() {
            return DAY_FORMATTER;
        }

        @Override
        public LocalDateTime tagToTime(String tag) {
            return LocalDate.parse(tag, this.formatter()).atStartOfDay();
        }
    }

    private static class HourlyTagPeriodHandler
    extends BaseTagPeriodHandler {
        private static final Duration ONE_PERIOD = Duration.ofHours(1L);

        private HourlyTagPeriodHandler() {
        }

        @Override
        protected Duration onePeriod() {
            return ONE_PERIOD;
        }

        @Override
        protected DateTimeFormatter formatter() {
            return HOUR_FORMATTER;
        }
    }

    private static abstract class BaseTagPeriodHandler
    implements TagPeriodHandler {
        private BaseTagPeriodHandler() {
        }

        protected abstract Duration onePeriod();

        protected abstract DateTimeFormatter formatter();

        @Override
        public void validateDelay(Duration delay) {
            Preconditions.checkArgument(this.onePeriod().compareTo(delay) > 0);
        }

        @Override
        public LocalDateTime tagToTime(String tag) {
            return LocalDateTime.parse(tag, this.formatter());
        }

        @Override
        public LocalDateTime normalizeToTagTime(LocalDateTime time) {
            long mills = Timestamp.fromLocalDateTime(time).getMillisecond();
            long periodMills = this.onePeriod().toMillis();
            LocalDateTime normalized = Timestamp.fromEpochMillis(mills / periodMills * periodMills).toLocalDateTime();
            return normalized.minus(this.onePeriod());
        }

        @Override
        public String timeToTag(LocalDateTime time) {
            return time.format(this.formatter());
        }

        @Override
        public LocalDateTime nextTagTime(LocalDateTime time) {
            return time.plus(this.onePeriod());
        }
    }

    private static interface TagPeriodHandler {
        public void validateDelay(Duration var1);

        public LocalDateTime tagToTime(String var1);

        public LocalDateTime normalizeToTagTime(LocalDateTime var1);

        public String timeToTag(LocalDateTime var1);

        public LocalDateTime nextTagTime(LocalDateTime var1);
    }

    private static class WatermarkExtractor
    implements TimeExtractor {
        private final ZoneId watermarkZoneId;

        private WatermarkExtractor(ZoneId watermarkZoneId) {
            this.watermarkZoneId = watermarkZoneId;
        }

        @Override
        public Optional<LocalDateTime> extract(Snapshot snapshot) {
            Long watermark = snapshot.watermark();
            if (watermark == null) {
                return Optional.empty();
            }
            return Optional.of(Instant.ofEpochMilli(watermark).atZone(this.watermarkZoneId).toLocalDateTime());
        }
    }

    private static class ProcessTimeExtractor
    implements TimeExtractor {
        private ProcessTimeExtractor() {
        }

        @Override
        public Optional<LocalDateTime> extract(Snapshot snapshot) {
            return Optional.of(Instant.ofEpochMilli(snapshot.timeMillis()).atZone(ZoneId.systemDefault()).toLocalDateTime());
        }
    }

    private static interface TimeExtractor {
        public Optional<LocalDateTime> extract(Snapshot var1);
    }
}

