/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import com.amazonaws.services.s3.model.CompleteMultipartUploadResult;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.BBPartHandle;
import org.apache.hadoop.fs.BBUploadHandle;
import org.apache.hadoop.fs.PartHandle;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathHandle;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.UploadHandle;
import org.apache.hadoop.fs.impl.AbstractMultipartUploader;
import org.apache.hadoop.fs.s3a.WriteOperations;
import org.apache.hadoop.fs.s3a.impl.S3AMultipartUploaderBuilder;
import org.apache.hadoop.fs.s3a.impl.StoreContext;
import org.apache.hadoop.fs.s3a.s3guard.BulkOperationState;
import org.apache.hadoop.fs.s3a.statistics.S3AMultipartUploaderStatistics;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsLogging;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Charsets;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class S3AMultipartUploader
extends AbstractMultipartUploader {
    private final S3AMultipartUploaderBuilder builder;
    public static final String HEADER = "S3A-part01";
    private final WriteOperations writeOperations;
    private final StoreContext context;
    private final S3AMultipartUploaderStatistics statistics;
    private BulkOperationState operationState;
    private boolean noOperationState;

    S3AMultipartUploader(S3AMultipartUploaderBuilder builder, WriteOperations writeOperations, StoreContext context, S3AMultipartUploaderStatistics statistics) {
        super(context.makeQualified(builder.getPath()));
        this.builder = builder;
        this.writeOperations = writeOperations;
        this.context = context;
        this.statistics = Objects.requireNonNull(statistics);
    }

    @Override
    public void close() throws IOException {
        if (this.operationState != null) {
            this.operationState.close();
        }
        super.close();
    }

    @Override
    public IOStatistics getIOStatistics() {
        return this.statistics.getIOStatistics();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("S3AMultipartUploader{");
        sb.append("base=").append(this.getBasePath());
        sb.append("; statistics=").append(IOStatisticsLogging.ioStatisticsToString(this.statistics.getIOStatistics()));
        sb.append('}');
        return sb.toString();
    }

    private synchronized BulkOperationState retrieveOperationState() throws IOException {
        if (this.operationState == null && !this.noOperationState) {
            this.operationState = this.writeOperations.initiateOperation(this.getBasePath(), BulkOperationState.OperationType.Upload);
            this.noOperationState = this.operationState != null;
        }
        return this.operationState;
    }

    @Override
    public CompletableFuture<UploadHandle> startUpload(Path filePath) throws IOException {
        Path dest = this.context.makeQualified(filePath);
        this.checkPath(dest);
        String key = this.context.pathToKey(dest);
        return this.context.submit(new CompletableFuture(), () -> {
            String uploadId = this.writeOperations.initiateMultiPartUpload(key);
            this.statistics.uploadStarted();
            return BBUploadHandle.from(ByteBuffer.wrap(uploadId.getBytes(Charsets.UTF_8)));
        });
    }

    @Override
    public CompletableFuture<PartHandle> putPart(UploadHandle uploadId, int partNumber, Path filePath, InputStream inputStream, long lengthInBytes) throws IOException {
        Path dest = this.context.makeQualified(filePath);
        this.checkPutArguments(dest, inputStream, partNumber, uploadId, lengthInBytes);
        byte[] uploadIdBytes = uploadId.toByteArray();
        this.checkUploadId(uploadIdBytes);
        String key = this.context.pathToKey(dest);
        String uploadIdString = new String(uploadIdBytes, 0, uploadIdBytes.length, Charsets.UTF_8);
        return this.context.submit(new CompletableFuture(), () -> {
            UploadPartRequest request = this.writeOperations.newUploadPartRequest(key, uploadIdString, partNumber, (int)lengthInBytes, inputStream, null, 0L);
            UploadPartResult result = this.writeOperations.uploadPart(request);
            this.statistics.partPut(lengthInBytes);
            String eTag = result.getETag();
            return BBPartHandle.from(ByteBuffer.wrap(S3AMultipartUploader.buildPartHandlePayload(filePath.toUri().toString(), uploadIdString, result.getPartNumber(), eTag, lengthInBytes)));
        });
    }

    @Override
    public CompletableFuture<PathHandle> complete(UploadHandle uploadHandle, Path filePath, Map<Integer, PartHandle> handleMap) throws IOException {
        Path dest = this.context.makeQualified(filePath);
        this.checkPath(dest);
        byte[] uploadIdBytes = uploadHandle.toByteArray();
        this.checkUploadId(uploadIdBytes);
        this.checkPartHandles(handleMap);
        ArrayList<Map.Entry<Integer, PartHandle>> handles = new ArrayList<Map.Entry<Integer, PartHandle>>(handleMap.entrySet());
        handles.sort(Comparator.comparingInt(Map.Entry::getKey));
        int count = handles.size();
        String key = this.context.pathToKey(dest);
        String uploadIdStr = new String(uploadIdBytes, 0, uploadIdBytes.length, Charsets.UTF_8);
        ArrayList<PartETag> eTags = new ArrayList<PartETag>();
        eTags.ensureCapacity(handles.size());
        long totalLength = 0L;
        HashSet<Integer> ids = new HashSet<Integer>(count);
        for (Map.Entry entry : handles) {
            PartHandlePayload payload = S3AMultipartUploader.parsePartHandlePayload(((PartHandle)entry.getValue()).toByteArray());
            payload.validate(uploadIdStr, filePath);
            ids.add(payload.getPartNumber());
            totalLength += payload.getLen();
            eTags.add(new PartETag((Integer)entry.getKey(), payload.getEtag()));
        }
        Preconditions.checkArgument(ids.size() == count, "Duplicate PartHandles");
        BulkOperationState state = this.retrieveOperationState();
        long l = totalLength;
        return this.context.submit(new CompletableFuture(), () -> {
            CompleteMultipartUploadResult result = this.writeOperations.commitUpload(key, uploadIdStr, eTags, finalLen, state);
            byte[] eTag = result.getETag().getBytes(Charsets.UTF_8);
            this.statistics.uploadCompleted();
            return () -> ByteBuffer.wrap(eTag);
        });
    }

    @Override
    public CompletableFuture<Void> abort(UploadHandle uploadId, Path filePath) throws IOException {
        Path dest = this.context.makeQualified(filePath);
        this.checkPath(dest);
        byte[] uploadIdBytes = uploadId.toByteArray();
        this.checkUploadId(uploadIdBytes);
        String uploadIdString = new String(uploadIdBytes, 0, uploadIdBytes.length, Charsets.UTF_8);
        return this.context.submit(new CompletableFuture(), () -> {
            this.writeOperations.abortMultipartCommit(this.context.pathToKey(dest), uploadIdString);
            this.statistics.uploadAborted();
            return null;
        });
    }

    @Override
    public CompletableFuture<Integer> abortUploadsUnderPath(Path path) throws IOException {
        this.statistics.abortUploadsUnderPathInvoked();
        return this.context.submit(new CompletableFuture(), () -> this.writeOperations.abortMultipartUploadsUnderPath(this.context.pathToKey(path)));
    }

    @VisibleForTesting
    static byte[] buildPartHandlePayload(String path, String uploadId, int partNumber, String etag, long len) throws IOException {
        return new PartHandlePayload(path, uploadId, partNumber, len, etag).toBytes();
    }

    @VisibleForTesting
    static PartHandlePayload parsePartHandlePayload(byte[] data) throws IOException {
        try (DataInputStream input = new DataInputStream(new ByteArrayInputStream(data));){
            String header = input.readUTF();
            if (!HEADER.equals(header)) {
                throw new IOException("Wrong header string: \"" + header + "\"");
            }
            String path = input.readUTF();
            String uploadId = input.readUTF();
            int partNumber = input.readInt();
            long len = input.readLong();
            String etag = input.readUTF();
            if (len < 0L) {
                throw new IOException("Negative length");
            }
            PartHandlePayload partHandlePayload = new PartHandlePayload(path, uploadId, partNumber, len, etag);
            return partHandlePayload;
        }
    }

    @VisibleForTesting
    static final class PartHandlePayload {
        private final String path;
        private final String uploadId;
        private final int partNumber;
        private final long len;
        private final String etag;

        private PartHandlePayload(String path, String uploadId, int partNumber, long len, String etag) {
            Preconditions.checkArgument(StringUtils.isNotEmpty(etag), "Empty etag");
            Preconditions.checkArgument(StringUtils.isNotEmpty(path), "Empty path");
            Preconditions.checkArgument(StringUtils.isNotEmpty(uploadId), "Empty uploadId");
            Preconditions.checkArgument(len >= 0L, "Invalid length");
            this.path = path;
            this.uploadId = uploadId;
            this.partNumber = partNumber;
            this.len = len;
            this.etag = etag;
        }

        public String getPath() {
            return this.path;
        }

        public int getPartNumber() {
            return this.partNumber;
        }

        public long getLen() {
            return this.len;
        }

        public String getEtag() {
            return this.etag;
        }

        public String getUploadId() {
            return this.uploadId;
        }

        public byte[] toBytes() throws IOException {
            Preconditions.checkArgument(StringUtils.isNotEmpty(this.etag), "Empty etag");
            Preconditions.checkArgument(this.len >= 0L, "Invalid length");
            ByteArrayOutputStream bytes = new ByteArrayOutputStream();
            try (DataOutputStream output = new DataOutputStream(bytes);){
                output.writeUTF(S3AMultipartUploader.HEADER);
                output.writeUTF(this.path);
                output.writeUTF(this.uploadId);
                output.writeInt(this.partNumber);
                output.writeLong(this.len);
                output.writeUTF(this.etag);
            }
            return bytes.toByteArray();
        }

        public void validate(String uploadIdStr, Path filePath) throws PathIOException {
            String destUri = filePath.toUri().toString();
            if (!destUri.equals(this.path)) {
                throw new PathIOException(destUri, "Multipart part path mismatch: " + this.path);
            }
            if (!uploadIdStr.equals(this.uploadId)) {
                throw new PathIOException(destUri, "Multipart part ID mismatch: " + this.uploadId);
            }
        }
    }
}

