/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.jobmaster.slotpool;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.clusterframework.types.ResourceProfile;
import org.apache.flink.runtime.jobmanager.slots.TaskManagerGateway;
import org.apache.flink.runtime.jobmaster.JobMasterId;
import org.apache.flink.runtime.jobmaster.RpcTaskManagerGateway;
import org.apache.flink.runtime.jobmaster.SlotInfo;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlot;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlotPool;
import org.apache.flink.runtime.jobmaster.slotpool.DefaultAllocatedSlotPool;
import org.apache.flink.runtime.jobmaster.slotpool.FreeSlotTracker;
import org.apache.flink.runtime.taskexecutor.TaskExecutorGateway;
import org.apache.flink.runtime.taskexecutor.TestingTaskExecutorGatewayBuilder;
import org.apache.flink.runtime.taskmanager.LocalTaskManagerLocation;
import org.apache.flink.runtime.taskmanager.TaskManagerLocation;
import org.apache.flink.shaded.guava33.com.google.common.collect.Iterables;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

class DefaultAllocatedSlotPoolTest {
    DefaultAllocatedSlotPoolTest() {
    }

    @Test
    void testAddSlots() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> slots = this.createAllocatedSlots();
        slotPool.addSlots(slots, 0L);
        this.assertSlotPoolContainsSlots(slotPool, slots);
        this.assertSlotPoolContainsFreeSlots(slotPool, slots);
    }

    @Test
    void testRemoveSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> slots = this.createAllocatedSlots();
        slotPool.addSlots(slots, 0L);
        Iterator<AllocatedSlot> iterator = slots.iterator();
        AllocatedSlot removedSlot = iterator.next();
        iterator.remove();
        slotPool.removeSlot(removedSlot.getAllocationId());
        this.assertSlotPoolContainsSlots(slotPool, slots);
    }

    @Test
    void testRemoveSlots() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        Collection<AllocatedSlot> slots = this.createAllocatedSlotsWithOwner(owner);
        AllocatedSlot otherSlot = this.createAllocatedSlot(ResourceID.generate());
        slots.add(otherSlot);
        slotPool.addSlots(slots, 0L);
        slotPool.removeSlots(owner);
        this.assertSlotPoolContainsSlots(slotPool, Collections.singleton(otherSlot));
    }

    @Test
    void testRemoveSlotsReturnValue() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        AllocatedSlot slot1 = this.createAllocatedSlot(owner);
        AllocatedSlot slot2 = this.createAllocatedSlot(owner);
        slotPool.addSlots(Arrays.asList(slot1, slot2), 0L);
        slotPool.reserveFreeSlot(slot1.getAllocationId());
        AllocatedSlotPool.AllocatedSlotsAndReservationStatus allocatedSlotsAndReservationStatus = slotPool.removeSlots(owner);
        Assertions.assertThat((Collection)allocatedSlotsAndReservationStatus.getAllocatedSlots()).containsExactlyInAnyOrder((Object[])new AllocatedSlot[]{slot1, slot2});
        Assertions.assertThat((boolean)allocatedSlotsAndReservationStatus.wasFree(slot1.getAllocationId())).isFalse();
        Assertions.assertThat((boolean)allocatedSlotsAndReservationStatus.wasFree(slot2.getAllocationId())).isTrue();
    }

    @Test
    void testContainsSlots() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(owner);
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        Assertions.assertThat((boolean)slotPool.containsSlots(owner)).isTrue();
        Assertions.assertThat((boolean)slotPool.containsSlots(ResourceID.generate())).isFalse();
    }

    @Test
    void testContainsSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(null);
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        Assertions.assertThat((boolean)slotPool.containsSlot(allocatedSlot.getAllocationId())).isTrue();
        Assertions.assertThat((boolean)slotPool.containsSlot(new AllocationID())).isFalse();
    }

    @Test
    void testReserveFreeSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> allSlots = this.createAllocatedSlots();
        ArrayList<AllocatedSlot> freeSlots = new ArrayList<AllocatedSlot>(allSlots);
        Iterator iterator = freeSlots.iterator();
        AllocatedSlot allocatedSlot = (AllocatedSlot)iterator.next();
        iterator.remove();
        slotPool.addSlots(allSlots, 0L);
        Assertions.assertThat((Object)slotPool.reserveFreeSlot(allocatedSlot.getAllocationId())).isEqualTo((Object)allocatedSlot);
        this.assertSlotPoolContainsFreeSlots(slotPool, freeSlots);
        this.assertSlotPoolContainsSlots(slotPool, allSlots);
    }

    @Test
    void testReserveNonFreeSlotFails() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot slot = this.createAllocatedSlot(null);
        slotPool.addSlots(Collections.singleton(slot), 0L);
        slotPool.reserveFreeSlot(slot.getAllocationId());
        Assertions.assertThatThrownBy(() -> slotPool.reserveFreeSlot(slot.getAllocationId())).isInstanceOf(IllegalStateException.class);
    }

    @Test
    void testFreeingOfReservedSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> slots = this.createAllocatedSlots();
        boolean initialTime = false;
        slotPool.addSlots(slots, 0L);
        AllocatedSlot slot = slots.iterator().next();
        slotPool.reserveFreeSlot(slot.getAllocationId());
        boolean releaseTime = true;
        Assertions.assertThat((Optional)slotPool.freeReservedSlot(slot.getAllocationId(), 1L)).isPresent();
        this.assertSlotPoolContainsFreeSlots(slotPool, slots);
        for (AllocatedSlotPool.FreeSlotInfo freeSlotInfo : slotPool.getFreeSlotTracker().getFreeSlotsWithIdleSinceInformation()) {
            long time = freeSlotInfo.getAllocationId().equals((Object)slot.getAllocationId()) ? 1L : 0L;
            Assertions.assertThat((long)freeSlotInfo.getFreeSince()).isEqualTo(time);
        }
    }

    @Test
    void testFreeingOfFreeSlotIsIgnored() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot slot = this.createAllocatedSlot(null);
        slotPool.addSlots(Collections.singleton(slot), 0L);
        Assertions.assertThat((Optional)slotPool.freeReservedSlot(slot.getAllocationId(), 1L)).isNotPresent();
        AllocatedSlotPool.FreeSlotInfo freeSlotInfo = (AllocatedSlotPool.FreeSlotInfo)Iterables.getOnlyElement((Iterable)slotPool.getFreeSlotTracker().getFreeSlotsWithIdleSinceInformation());
        Assertions.assertThat((long)freeSlotInfo.getFreeSince()).isEqualTo(0L);
    }

    @Test
    void testSlotUtilizationCalculation() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        Collection<AllocatedSlot> slots = this.createAllocatedSlotsWithOwner(owner);
        slotPool.addSlots(slots, 0L);
        FreeSlotTracker freeSlotTracker = slotPool.getFreeSlotTracker();
        Assertions.assertThat((Collection)freeSlotTracker.getAvailableSlots()).allSatisfy(allocationId -> Assertions.assertThat((double)freeSlotTracker.getTaskExecutorUtilization(freeSlotTracker.getSlotInfo(allocationId))).isCloseTo(0.0, Assertions.offset((Double)0.1)));
        int numAllocatedSlots = 0;
        for (AllocatedSlot slot : slots) {
            Assertions.assertThat((Object)slotPool.reserveFreeSlot(slot.getAllocationId())).isEqualTo((Object)slot);
            freeSlotTracker.reserveSlot(slot.getAllocationId());
            double utilization = (double)(++numAllocatedSlots) / (double)slots.size();
            Assertions.assertThat((Collection)freeSlotTracker.getAvailableSlots()).allSatisfy(allocationId -> Assertions.assertThat((double)freeSlotTracker.getTaskExecutorUtilization(freeSlotTracker.getSlotInfo(allocationId))).isCloseTo(utilization, Assertions.offset((Double)0.1)));
        }
    }

    @Test
    void testRemoveSlotsOfUnknownOwnerIsIgnored() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        slotPool.removeSlots(ResourceID.generate());
    }

    @Test
    void testContainsFreeSlotReturnsTrueIfSlotIsFree() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(ResourceID.generate());
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        Assertions.assertThat((boolean)slotPool.containsFreeSlot(allocatedSlot.getAllocationId())).isTrue();
    }

    @Test
    void testContainsFreeSlotReturnsFalseIfSlotDoesNotExist() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Assertions.assertThat((boolean)slotPool.containsFreeSlot(new AllocationID())).isFalse();
    }

    @Test
    void testContainsFreeSlotReturnsFalseIfSlotIsReserved() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(ResourceID.generate());
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        slotPool.reserveFreeSlot(allocatedSlot.getAllocationId());
        Assertions.assertThat((boolean)slotPool.containsFreeSlot(allocatedSlot.getAllocationId())).isFalse();
    }

    private void assertSlotPoolContainsSlots(DefaultAllocatedSlotPool slotPool, Collection<AllocatedSlot> slots) {
        Assertions.assertThat((Collection)slotPool.getAllSlotsInformation()).hasSize(slots.size());
        Map slotsPerAllocationId = slots.stream().collect(Collectors.toMap(AllocatedSlot::getAllocationId, Function.identity()));
        Assertions.assertThat((Collection)slotPool.getAllSlotsInformation()).allSatisfy(slotInfo -> Assertions.assertThat((Object)((AllocatedSlot)slotsPerAllocationId.get(slotInfo.getAllocationId()))).isEqualTo(slotInfo));
    }

    private void assertSlotPoolContainsFreeSlots(DefaultAllocatedSlotPool slotPool, Collection<AllocatedSlot> allocatedSlots) {
        Collection freeSlotsInformation = slotPool.getFreeSlotTracker().getFreeSlotsWithIdleSinceInformation();
        Assertions.assertThat((Collection)freeSlotsInformation).hasSize(allocatedSlots.size());
        Map allocatedSlotMap = allocatedSlots.stream().collect(Collectors.toMap(AllocatedSlot::getAllocationId, Function.identity()));
        Assertions.assertThat((Collection)freeSlotsInformation).allSatisfy(freeSlotInfo -> {
            AllocatedSlot allocatedSlot = (AllocatedSlot)allocatedSlotMap.get(freeSlotInfo.getAllocationId());
            Assertions.assertThat((Object)allocatedSlot).isNotNull();
            SlotInfo slotInfo = freeSlotInfo.asSlotInfo();
            Assertions.assertThat((Comparable)allocatedSlot.getAllocationId()).isEqualTo((Object)slotInfo.getAllocationId());
            Assertions.assertThat((int)allocatedSlot.getPhysicalSlotNumber()).isEqualTo(slotInfo.getPhysicalSlotNumber());
            Assertions.assertThat((Object)allocatedSlot.getResourceProfile()).isEqualTo((Object)slotInfo.getResourceProfile());
            Assertions.assertThat((Comparable)allocatedSlot.getTaskManagerLocation()).isEqualTo((Object)slotInfo.getTaskManagerLocation());
        });
    }

    private Collection<AllocatedSlot> createAllocatedSlots() {
        return new ArrayList<AllocatedSlot>(Arrays.asList(this.createAllocatedSlot(null), this.createAllocatedSlot(null), this.createAllocatedSlot(null)));
    }

    private Collection<AllocatedSlot> createAllocatedSlotsWithOwner(ResourceID owner) {
        return new ArrayList<AllocatedSlot>(Arrays.asList(this.createAllocatedSlot(owner), this.createAllocatedSlot(owner), this.createAllocatedSlot(owner)));
    }

    @Nonnull
    private AllocatedSlot createAllocatedSlot(@Nullable ResourceID owner) {
        return new AllocatedSlot(new AllocationID(), (TaskManagerLocation)(owner == null ? new LocalTaskManagerLocation() : new TaskManagerLocation(owner, InetAddress.getLoopbackAddress(), 41)), 0, ResourceProfile.UNKNOWN, (TaskManagerGateway)new RpcTaskManagerGateway((TaskExecutorGateway)new TestingTaskExecutorGatewayBuilder().createTestingTaskExecutorGateway(), JobMasterId.generate()));
    }
}

