/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.UUID;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferBuilderTestUtils;
import org.apache.flink.runtime.io.network.buffer.BufferRecycler;
import org.apache.flink.runtime.io.network.buffer.FreeingBufferRecycler;
import org.apache.flink.runtime.io.network.partition.BufferReaderWriterUtil;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

class BufferReaderWriterUtilTest {
    BufferReaderWriterUtilTest() {
    }

    @Test
    void writeReadByteBuffer() {
        ByteBuffer memory = ByteBuffer.allocateDirect(1200);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        BufferReaderWriterUtil.configureByteBuffer((ByteBuffer)memory);
        BufferReaderWriterUtil.writeBuffer((Buffer)buffer, (ByteBuffer)memory);
        int pos = memory.position();
        memory.flip();
        Buffer result = BufferReaderWriterUtil.sliceNextBuffer((ByteBuffer)memory);
        Assertions.assertThat((int)memory.position()).isEqualTo(pos);
        BufferReaderWriterUtilTest.validateTestBuffer(result);
    }

    @Test
    void writeByteBufferNotEnoughSpace() {
        ByteBuffer memory = ByteBuffer.allocateDirect(10);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        boolean written = BufferReaderWriterUtil.writeBuffer((Buffer)buffer, (ByteBuffer)memory);
        Assertions.assertThat((boolean)written).isFalse();
        Assertions.assertThat((int)memory.position()).isZero();
        Assertions.assertThat((int)memory.limit()).isEqualTo(memory.capacity());
    }

    @Test
    void readFromEmptyByteBuffer() {
        ByteBuffer memory = ByteBuffer.allocateDirect(100);
        memory.position(memory.limit());
        Buffer result = BufferReaderWriterUtil.sliceNextBuffer((ByteBuffer)memory);
        Assertions.assertThat((Object)result).isNull();
    }

    @Test
    void testReadFromByteBufferNotEnoughData() {
        ByteBuffer memory = ByteBuffer.allocateDirect(1200);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        BufferReaderWriterUtil.writeBuffer((Buffer)buffer, (ByteBuffer)memory);
        memory.flip().limit(memory.limit() - 1);
        ByteBuffer tooSmall = memory.slice();
        Assertions.assertThatThrownBy(() -> BufferReaderWriterUtil.sliceNextBuffer((ByteBuffer)tooSmall)).isInstanceOf(IllegalArgumentException.class);
    }

    @Test
    void writeReadFileChannel(@TempDir Path tempPath) throws Exception {
        FileChannel fc = BufferReaderWriterUtilTest.tmpFileChannel(tempPath);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        MemorySegment readBuffer = MemorySegmentFactory.allocateUnpooledOffHeapMemory((int)buffer.getSize(), null);
        BufferReaderWriterUtil.writeToByteChannel((FileChannel)fc, (Buffer)buffer, (ByteBuffer[])BufferReaderWriterUtil.allocatedWriteBufferArray());
        fc.position(0L);
        Buffer result = BufferReaderWriterUtil.readFromByteChannel((FileChannel)fc, (ByteBuffer)BufferReaderWriterUtil.allocatedHeaderBuffer(), (MemorySegment)readBuffer, (BufferRecycler)FreeingBufferRecycler.INSTANCE);
        BufferReaderWriterUtilTest.validateTestBuffer(result);
    }

    @Test
    void readPrematureEndOfFile1(@TempDir Path tempPath) throws Exception {
        FileChannel fc = BufferReaderWriterUtilTest.tmpFileChannel(tempPath);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        MemorySegment readBuffer = MemorySegmentFactory.allocateUnpooledOffHeapMemory((int)buffer.getSize(), null);
        BufferReaderWriterUtil.writeToByteChannel((FileChannel)fc, (Buffer)buffer, (ByteBuffer[])BufferReaderWriterUtil.allocatedWriteBufferArray());
        fc.truncate(fc.position() - 1L);
        fc.position(0L);
        Assertions.assertThatThrownBy(() -> BufferReaderWriterUtil.readFromByteChannel((FileChannel)fc, (ByteBuffer)BufferReaderWriterUtil.allocatedHeaderBuffer(), (MemorySegment)readBuffer, (BufferRecycler)FreeingBufferRecycler.INSTANCE)).isInstanceOf(IOException.class);
    }

    @Test
    void readPrematureEndOfFile2(@TempDir Path tempPath) throws Exception {
        FileChannel fc = BufferReaderWriterUtilTest.tmpFileChannel(tempPath);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        MemorySegment readBuffer = MemorySegmentFactory.allocateUnpooledOffHeapMemory((int)buffer.getSize(), null);
        BufferReaderWriterUtil.writeToByteChannel((FileChannel)fc, (Buffer)buffer, (ByteBuffer[])BufferReaderWriterUtil.allocatedWriteBufferArray());
        fc.truncate(2L);
        fc.position(0L);
        Assertions.assertThatThrownBy(() -> BufferReaderWriterUtil.readFromByteChannel((FileChannel)fc, (ByteBuffer)BufferReaderWriterUtil.allocatedHeaderBuffer(), (MemorySegment)readBuffer, (BufferRecycler)FreeingBufferRecycler.INSTANCE)).isInstanceOf(IOException.class);
    }

    @Test
    void testBulkWritingLargeNumberOfBuffers(@TempDir Path tempPath) throws Exception {
        int bufferSize = 1024;
        int numBuffers = 1025;
        try (FileChannel fileChannel = BufferReaderWriterUtilTest.tmpFileChannel(tempPath);){
            ByteBuffer[] data = new ByteBuffer[numBuffers];
            for (int i = 0; i < numBuffers; ++i) {
                data[i] = ByteBuffer.allocateDirect(bufferSize);
            }
            int bytesExpected = bufferSize * numBuffers;
            BufferReaderWriterUtil.writeBuffers((FileChannel)fileChannel, (long)bytesExpected, (ByteBuffer[])data);
            Assertions.assertThat((long)fileChannel.size()).isEqualTo((long)bytesExpected);
        }
    }

    @Test
    void testPositionToNextBuffer(@TempDir Path tempPath) throws Exception {
        FileChannel fc = BufferReaderWriterUtilTest.tmpFileChannel(tempPath);
        ByteBuffer[] byteBuffersWithHeader = BufferReaderWriterUtilTest.createByteBuffersWithHeader(2);
        long totalBytes = Arrays.stream(byteBuffersWithHeader).mapToLong(java.nio.Buffer::remaining).sum();
        BufferReaderWriterUtil.writeBuffers((FileChannel)fc, (long)totalBytes, (ByteBuffer[])byteBuffersWithHeader);
        fc.position(0L);
        BufferReaderWriterUtil.positionToNextBuffer((FileChannel)fc, (ByteBuffer)byteBuffersWithHeader[0]);
        long expectedPosition = totalBytes / 2L;
        Assertions.assertThat((long)fc.position()).isEqualTo(expectedPosition);
    }

    @Test
    void writeFileReadMemoryBuffer(@TempDir Path tempPath) throws Exception {
        FileChannel fc = BufferReaderWriterUtilTest.tmpFileChannel(tempPath);
        Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
        BufferReaderWriterUtil.writeToByteChannel((FileChannel)fc, (Buffer)buffer, (ByteBuffer[])BufferReaderWriterUtil.allocatedWriteBufferArray());
        ByteBuffer bb = fc.map(FileChannel.MapMode.READ_ONLY, 0L, fc.position()).order(ByteOrder.nativeOrder());
        BufferReaderWriterUtil.configureByteBuffer((ByteBuffer)bb);
        fc.close();
        Buffer result = BufferReaderWriterUtil.sliceNextBuffer((ByteBuffer)bb);
        BufferReaderWriterUtilTest.validateTestBuffer(result);
    }

    private static FileChannel tmpFileChannel(Path tempPath) throws IOException {
        return FileChannel.open(Files.createFile(tempPath.resolve(UUID.randomUUID().toString()), new FileAttribute[0]), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
    }

    private static ByteBuffer[] createByteBuffersWithHeader(int numBuffers) {
        ByteBuffer[] buffers = new ByteBuffer[numBuffers * 2];
        for (int i = 0; i < numBuffers; ++i) {
            buffers[2 * i] = BufferReaderWriterUtil.allocatedHeaderBuffer();
            Buffer buffer = BufferReaderWriterUtilTest.createTestBuffer();
            BufferReaderWriterUtil.setByteChannelBufferHeader((Buffer)buffer, (ByteBuffer)buffers[2 * i]);
            buffers[2 * i + 1] = buffer.getNioBufferReadable();
        }
        return buffers;
    }

    private static Buffer createTestBuffer() {
        return BufferBuilderTestUtils.buildBufferWithAscendingInts(1024, 200, 0);
    }

    private static void validateTestBuffer(Buffer buffer) {
        BufferBuilderTestUtils.validateBufferWithAscendingInts(buffer, 200, 0);
    }
}

