/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.partition.BufferReaderWriterUtil;
import org.apache.flink.runtime.io.network.partition.BufferWithSubpartition;
import org.apache.flink.runtime.io.network.partition.PartitionedFile;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.IOUtils;
import org.apache.flink.util.Preconditions;

@NotThreadSafe
public class PartitionedFileWriter
implements AutoCloseable {
    private static final int MIN_INDEX_BUFFER_SIZE = 800;
    private final int numSubpartitions;
    private final FileChannel dataFileChannel;
    private final FileChannel indexFileChannel;
    private final Path dataFilePath;
    private final Path indexFilePath;
    private final long[] subpartitionOffsets;
    private final long[] subpartitionBytes;
    private final int maxIndexBufferSize;
    private ByteBuffer indexBuffer;
    private boolean allIndexEntriesCached = true;
    private long totalBytesWritten;
    private int numRegions;
    private long numBuffers;
    private int currentSubpartition = -1;
    private boolean isBroadcastRegion;
    private boolean isFinished;
    private boolean isClosed;
    private final int[] writeOrder;
    private long preRegionTotalBytesWritten;

    public PartitionedFileWriter(int numSubpartitions, int maxIndexBufferSize, String basePath, int[] writeOrder) throws IOException {
        this(numSubpartitions, 800, maxIndexBufferSize, basePath, writeOrder);
    }

    @VisibleForTesting
    PartitionedFileWriter(int numSubpartitions, int minIndexBufferSize, int maxIndexBufferSize, String basePath, int[] writeOrder) throws IOException {
        Preconditions.checkArgument((numSubpartitions > 0 ? 1 : 0) != 0, (Object)"Illegal number of subpartitions.");
        Preconditions.checkArgument((maxIndexBufferSize > 0 ? 1 : 0) != 0, (Object)"Illegal maximum index cache size.");
        Preconditions.checkArgument((basePath != null ? 1 : 0) != 0, (Object)"Base path must not be null.");
        this.numSubpartitions = numSubpartitions;
        this.maxIndexBufferSize = this.alignMaxIndexBufferSize(maxIndexBufferSize);
        this.subpartitionOffsets = new long[numSubpartitions];
        this.subpartitionBytes = new long[numSubpartitions];
        this.dataFilePath = new File(basePath + ".shuffle.data").toPath();
        this.indexFilePath = new File(basePath + ".shuffle.index").toPath();
        this.writeOrder = (int[])Preconditions.checkNotNull((Object)writeOrder);
        this.indexBuffer = ByteBuffer.allocate(minIndexBufferSize);
        BufferReaderWriterUtil.configureByteBuffer(this.indexBuffer);
        this.dataFileChannel = this.openFileChannel(this.dataFilePath);
        try {
            this.indexFileChannel = this.openFileChannel(this.indexFilePath);
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly((AutoCloseable)this.dataFileChannel);
            IOUtils.deleteFileQuietly((Path)this.dataFilePath);
            throw throwable;
        }
    }

    private FileChannel openFileChannel(Path path) throws IOException {
        return FileChannel.open(path, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
    }

    private int alignMaxIndexBufferSize(int maxIndexBufferSize) {
        return maxIndexBufferSize / 16 * 16;
    }

    public void startNewRegion(boolean isBroadcastRegion) throws IOException {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"File writer is already finished.");
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"File writer is already closed.");
        this.writeRegionIndex();
        this.isBroadcastRegion = isBroadcastRegion;
    }

    private void writeIndexEntry(long subpartitionOffset, long numBytes) throws IOException {
        if (!this.indexBuffer.hasRemaining() && !this.extendIndexBufferIfPossible()) {
            this.flushIndexBuffer();
            this.indexBuffer.clear();
            this.allIndexEntriesCached = false;
        }
        this.indexBuffer.putLong(subpartitionOffset);
        this.indexBuffer.putLong(numBytes);
    }

    private boolean extendIndexBufferIfPossible() {
        if (this.indexBuffer.capacity() >= this.maxIndexBufferSize) {
            return false;
        }
        int newIndexBufferSize = Math.min(this.maxIndexBufferSize, 2 * this.indexBuffer.capacity());
        ByteBuffer newIndexBuffer = ByteBuffer.allocate(newIndexBufferSize);
        this.indexBuffer.flip();
        newIndexBuffer.put(this.indexBuffer);
        BufferReaderWriterUtil.configureByteBuffer(newIndexBuffer);
        this.indexBuffer = newIndexBuffer;
        return true;
    }

    private void writeRegionIndex() throws IOException {
        if (Arrays.stream(this.subpartitionBytes).sum() > 0L) {
            this.updateEmptySubpartitionOffsets();
            for (int subpartition = 0; subpartition < this.numSubpartitions; ++subpartition) {
                this.writeIndexEntry(this.subpartitionOffsets[subpartition], this.subpartitionBytes[subpartition]);
            }
            this.currentSubpartition = -1;
            ++this.numRegions;
            Arrays.fill(this.subpartitionBytes, 0L);
            this.preRegionTotalBytesWritten = this.totalBytesWritten;
        }
    }

    private void updateEmptySubpartitionOffsets() {
        for (int i = 0; i < this.writeOrder.length; ++i) {
            int currentSubPartition = this.writeOrder[i];
            if (this.subpartitionBytes[currentSubPartition] != 0L) continue;
            if (i == 0) {
                this.subpartitionOffsets[currentSubPartition] = this.preRegionTotalBytesWritten;
                continue;
            }
            int preSubPartition = this.writeOrder[i - 1];
            this.subpartitionOffsets[currentSubPartition] = this.subpartitionOffsets[preSubPartition] + this.subpartitionBytes[preSubPartition];
        }
    }

    private void flushIndexBuffer() throws IOException {
        this.indexBuffer.flip();
        if (this.indexBuffer.limit() > 0) {
            BufferReaderWriterUtil.writeBuffer(this.indexFileChannel, this.indexBuffer);
        }
    }

    public void writeBuffers(List<BufferWithSubpartition> bufferWithSubpartitions) throws IOException {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"File writer is already finished.");
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"File writer is already closed.");
        if (bufferWithSubpartitions.isEmpty()) {
            return;
        }
        this.numBuffers += (long)bufferWithSubpartitions.size();
        ByteBuffer[] bufferWithHeaders = new ByteBuffer[2 * bufferWithSubpartitions.size()];
        long expectedBytes = this.isBroadcastRegion ? this.collectBroadcastBuffers(bufferWithSubpartitions, bufferWithHeaders) : this.collectUnicastBuffers(bufferWithSubpartitions, bufferWithHeaders);
        this.totalBytesWritten += expectedBytes;
        BufferReaderWriterUtil.writeBuffers(this.dataFileChannel, expectedBytes, bufferWithHeaders);
    }

    private long collectUnicastBuffers(List<BufferWithSubpartition> bufferWithSubpartitions, ByteBuffer[] bufferWithHeaders) {
        long expectedBytes = 0L;
        long fileOffset = this.totalBytesWritten;
        for (int i = 0; i < bufferWithSubpartitions.size(); ++i) {
            int subpartition = bufferWithSubpartitions.get(i).getSubpartitionIndex();
            if (subpartition != this.currentSubpartition) {
                Preconditions.checkState((this.subpartitionBytes[subpartition] == 0L ? 1 : 0) != 0, (Object)"Must write data of the same subpartition together.");
                this.subpartitionOffsets[subpartition] = fileOffset;
                this.currentSubpartition = subpartition;
            }
            Buffer buffer = bufferWithSubpartitions.get(i).getBuffer();
            int numBytes = this.setBufferWithHeader(buffer, bufferWithHeaders, 2 * i);
            expectedBytes += (long)numBytes;
            fileOffset += (long)numBytes;
            int n = subpartition;
            this.subpartitionBytes[n] = this.subpartitionBytes[n] + (long)numBytes;
        }
        return expectedBytes;
    }

    private long collectBroadcastBuffers(List<BufferWithSubpartition> bufferWithSubpartitions, ByteBuffer[] bufferWithHeaders) {
        if (this.subpartitionBytes[0] == 0L) {
            for (int subpartition = 0; subpartition < this.numSubpartitions; ++subpartition) {
                this.subpartitionOffsets[subpartition] = this.totalBytesWritten;
            }
        }
        long expectedBytes = 0L;
        for (int i = 0; i < bufferWithSubpartitions.size(); ++i) {
            Buffer buffer = bufferWithSubpartitions.get(i).getBuffer();
            int numBytes = this.setBufferWithHeader(buffer, bufferWithHeaders, 2 * i);
            expectedBytes += (long)numBytes;
        }
        int subpartition = 0;
        while (subpartition < this.numSubpartitions) {
            int n = subpartition++;
            this.subpartitionBytes[n] = this.subpartitionBytes[n] + expectedBytes;
        }
        return expectedBytes;
    }

    private int setBufferWithHeader(Buffer buffer, ByteBuffer[] bufferWithHeaders, int index) {
        ByteBuffer header = BufferReaderWriterUtil.allocatedHeaderBuffer();
        BufferReaderWriterUtil.setByteChannelBufferHeader(buffer, header);
        bufferWithHeaders[index] = header;
        bufferWithHeaders[index + 1] = buffer.getNioBufferReadable();
        return header.remaining() + buffer.readableBytes();
    }

    public PartitionedFile finish() throws IOException {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"File writer is already finished.");
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"File writer is already closed.");
        this.isFinished = true;
        this.writeRegionIndex();
        this.flushIndexBuffer();
        this.indexBuffer.rewind();
        long dataFileSize = this.dataFileChannel.size();
        long indexFileSize = this.indexFileChannel.size();
        this.close();
        ByteBuffer indexEntryCache = null;
        if (this.allIndexEntriesCached) {
            indexEntryCache = this.indexBuffer;
        }
        this.indexBuffer = null;
        return new PartitionedFile(this.numRegions, this.numSubpartitions, this.dataFilePath, this.indexFilePath, dataFileSize, indexFileSize, this.numBuffers, indexEntryCache);
    }

    public void releaseQuietly() {
        IOUtils.closeQuietly((AutoCloseable)this);
        IOUtils.deleteFileQuietly((Path)this.dataFilePath);
        IOUtils.deleteFileQuietly((Path)this.indexFilePath);
    }

    @Override
    public void close() throws IOException {
        if (this.isClosed) {
            return;
        }
        this.isClosed = true;
        IOException exception = null;
        try {
            this.dataFileChannel.close();
        }
        catch (IOException ioException) {
            exception = ioException;
        }
        try {
            this.indexFileChannel.close();
        }
        catch (IOException ioException) {
            exception = (IOException)ExceptionUtils.firstOrSuppressed((Throwable)ioException, (Throwable)exception);
        }
        if (exception != null) {
            throw exception;
        }
    }
}

