/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.memory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.runtime.jobgraph.tasks.AbstractInvokable;
import org.apache.flink.runtime.memory.MemoryAllocationException;
import org.apache.flink.runtime.memory.MemoryManager;
import org.apache.flink.runtime.memory.MemoryManagerBuilder;
import org.apache.flink.runtime.memory.MemoryReservationException;
import org.apache.flink.runtime.operators.testutils.DummyInvokable;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ThrowableTypeAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class MemoryManagerTest {
    private static final long RANDOM_SEED = 643196033469871L;
    private static final int MEMORY_SIZE = 0x4800000;
    private static final int PAGE_SIZE = 32768;
    private static final int NUM_PAGES = 2304;
    private MemoryManager memoryManager;
    private Random random;

    MemoryManagerTest() {
    }

    @BeforeEach
    void setUp() {
        this.memoryManager = MemoryManagerBuilder.newBuilder().setMemorySize(0x4800000L).setPageSize(32768).build();
        this.random = new Random(643196033469871L);
    }

    @AfterEach
    void tearDown() {
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.memoryManager.verifyEmpty()).as("Memory manager is not complete empty and valid at the end of the test.", new Object[0])).isTrue();
        this.memoryManager = null;
        this.random = null;
    }

    @Test
    void allocateAllSingle() {
        DummyInvokable mockInvoke = new DummyInvokable();
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>();
        try {
            for (int i = 0; i < 2304; ++i) {
                segments.add((MemorySegment)this.memoryManager.allocatePages((Object)mockInvoke, 1).get(0));
            }
        }
        catch (MemoryAllocationException e) {
            Assertions.fail((String)"Unable to allocate memory");
        }
        this.memoryManager.release(segments);
    }

    @Test
    void allocateAllMulti() {
        DummyInvokable mockInvoke = new DummyInvokable();
        ArrayList segments = new ArrayList();
        try {
            for (int i = 0; i < 1152; ++i) {
                segments.addAll(this.memoryManager.allocatePages((Object)mockInvoke, 2));
            }
        }
        catch (MemoryAllocationException e) {
            Assertions.fail((String)"Unable to allocate memory");
        }
        this.memoryManager.release(segments);
    }

    @Test
    void allocateMultipleOwners() throws Exception {
        int i;
        int numOwners = 17;
        AbstractInvokable[] owners = new AbstractInvokable[17];
        List[] mems = new List[17];
        for (i = 0; i < 17; ++i) {
            owners[i] = new DummyInvokable();
            mems[i] = new ArrayList(64);
        }
        for (i = 0; i < 2304; ++i) {
            int owner = this.random.nextInt(17);
            mems[owner].addAll(this.memoryManager.allocatePages((Object)owners[owner], 1));
        }
        for (i = 0; i < 17; ++i) {
            this.memoryManager.releaseAll((Object)owners[i]);
            owners[i] = null;
            ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.allMemorySegmentsFreed(mems[i])).as("Released memory segments have not been destroyed.", new Object[0])).isTrue();
            mems[i] = null;
            for (int k = i + 1; k < 17; ++k) {
                ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.allMemorySegmentsValid(mems[k])).as("Non-released memory segments are accidentaly destroyed.", new Object[0])).isTrue();
            }
        }
    }

    @Test
    void allocateTooMuch() throws Exception {
        DummyInvokable mockInvoke = new DummyInvokable();
        List segs = this.memoryManager.allocatePages((Object)mockInvoke, 2304);
        this.testCannotAllocateAnymore((Object)mockInvoke, 1);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.allMemorySegmentsValid(segs)).as("The previously allocated segments were not valid any more.", new Object[0])).isTrue();
        this.memoryManager.releaseAll((Object)mockInvoke);
    }

    @Test
    void doubleReleaseReturnsMemoryOnlyOnce() throws MemoryAllocationException {
        DummyInvokable mockInvoke = new DummyInvokable();
        List segs = this.memoryManager.allocatePages((Object)mockInvoke, 2304);
        MemorySegment segment = (MemorySegment)segs.iterator().next();
        this.memoryManager.release(segment);
        this.memoryManager.release(segment);
        this.testCannotAllocateAnymore((Object)mockInvoke, 2);
        this.memoryManager.releaseAll((Object)mockInvoke);
    }

    @Test
    void releaseCollectionAfterReleaseAll() throws MemoryAllocationException {
        DummyInvokable mockInvoke = new DummyInvokable();
        List segs = this.memoryManager.allocatePages((Object)mockInvoke, 1);
        MemorySegment segment = (MemorySegment)segs.iterator().next();
        this.memoryManager.releaseAll((Object)mockInvoke);
        this.memoryManager.release(new ArrayList<MemorySegment>(Collections.singletonList(segment)));
    }

    @Test
    void releaseAfterReleaseAll() throws MemoryAllocationException {
        DummyInvokable mockInvoke = new DummyInvokable();
        List segs = this.memoryManager.allocatePages((Object)mockInvoke, 1);
        MemorySegment segment = (MemorySegment)segs.iterator().next();
        this.memoryManager.releaseAll((Object)mockInvoke);
        this.memoryManager.release(segment);
    }

    @Test
    void releaseSameSegmentFromTwoCollections() throws MemoryAllocationException {
        DummyInvokable mockInvoke = new DummyInvokable();
        MemorySegment seg1 = (MemorySegment)this.memoryManager.allocatePages((Object)mockInvoke, 1).get(0);
        MemorySegment seg2 = (MemorySegment)this.memoryManager.allocatePages((Object)mockInvoke, 1).get(0);
        MemorySegment seg3 = (MemorySegment)this.memoryManager.allocatePages((Object)mockInvoke, 1).get(0);
        this.memoryManager.release(new ArrayList<MemorySegment>(Arrays.asList(seg1, seg2)));
        this.memoryManager.release(new ArrayList<MemorySegment>(Arrays.asList(seg2, seg3)));
    }

    private boolean allMemorySegmentsValid(List<MemorySegment> memSegs) {
        for (MemorySegment seg : memSegs) {
            if (!seg.isFreed()) continue;
            return false;
        }
        return true;
    }

    private boolean allMemorySegmentsFreed(List<MemorySegment> memSegs) {
        for (MemorySegment seg : memSegs) {
            if (seg.isFreed()) continue;
            return false;
        }
        return true;
    }

    @Test
    void testMemoryReservation() throws MemoryReservationException {
        Object owner = new Object();
        this.memoryManager.reserveMemory(owner, 32768L);
        this.memoryManager.releaseMemory(owner, 32768L);
    }

    @Test
    void testAllMemoryReservation() throws MemoryReservationException {
        Object owner = new Object();
        this.memoryManager.reserveMemory(owner, this.memoryManager.getMemorySize());
        this.memoryManager.releaseAllMemory(owner);
    }

    @Test
    void testCannotReserveBeyondTheLimit() throws MemoryReservationException {
        Object owner = new Object();
        this.memoryManager.reserveMemory(owner, this.memoryManager.getMemorySize());
        this.testCannotReserveAnymore(1L);
        this.memoryManager.releaseAllMemory(owner);
    }

    @Test
    void testMemoryTooBigReservation() {
        long size = this.memoryManager.getMemorySize() + 32768L;
        this.testCannotReserveAnymore(size);
    }

    @Test
    void testMemoryReleaseMultipleTimes() throws MemoryReservationException {
        Object owner = new Object();
        Object owner2 = new Object();
        long totalHeapMemorySize = this.memoryManager.availableMemory();
        this.memoryManager.reserveMemory(owner2, 32768L);
        this.memoryManager.reserveMemory(owner, 32768L);
        this.memoryManager.releaseMemory(owner, 32768L);
        this.memoryManager.releaseMemory(owner, 32768L);
        long heapMemoryLeft = this.memoryManager.availableMemory();
        ((AbstractLongAssert)Assertions.assertThat((long)heapMemoryLeft).as("Memory leak happens", new Object[0])).isEqualTo(totalHeapMemorySize - 32768L);
        this.memoryManager.releaseAllMemory(owner2);
    }

    @Test
    void testMemoryReleaseMoreThanReserved() throws MemoryReservationException {
        Object owner = new Object();
        Object owner2 = new Object();
        long totalHeapMemorySize = this.memoryManager.availableMemory();
        this.memoryManager.reserveMemory(owner2, 32768L);
        this.memoryManager.reserveMemory(owner, 32768L);
        this.memoryManager.releaseMemory(owner, 65536L);
        long heapMemoryLeft = this.memoryManager.availableMemory();
        ((AbstractLongAssert)Assertions.assertThat((long)heapMemoryLeft).as("Memory leak happens", new Object[0])).isEqualTo(totalHeapMemorySize - 32768L);
        this.memoryManager.releaseAllMemory(owner2);
    }

    @Test
    void testMemoryAllocationAndReservation() throws MemoryAllocationException, MemoryReservationException {
        int totalPagesForType = (int)this.memoryManager.getMemorySize() / 32768;
        Object owner1 = new Object();
        this.memoryManager.allocatePages(owner1, totalPagesForType / 2);
        Object owner2 = new Object();
        this.memoryManager.reserveMemory(owner2, 32768L * (long)totalPagesForType / 2L);
        this.testCannotAllocateAnymore(new Object(), 1);
        this.testCannotReserveAnymore(1L);
        this.memoryManager.releaseAll(owner1);
        this.memoryManager.releaseAllMemory(owner2);
    }

    @Test
    void testComputeMemorySize() {
        double fraction = 0.6;
        Assertions.assertThat((long)this.memoryManager.computeMemorySize(fraction)).isEqualTo((long)((double)this.memoryManager.getMemorySize() * fraction));
        fraction = 1.0;
        Assertions.assertThat((long)this.memoryManager.computeMemorySize(fraction)).isEqualTo((long)((double)this.memoryManager.getMemorySize() * fraction));
    }

    @Test
    void testComputeMemorySizeFailForZeroFraction() {
        Assertions.assertThatExceptionOfType(IllegalArgumentException.class).isThrownBy(() -> this.memoryManager.computeMemorySize(0.0));
    }

    @Test
    void testComputeMemorySizeFailForTooLargeFraction() {
        Assertions.assertThatExceptionOfType(IllegalArgumentException.class).isThrownBy(() -> this.memoryManager.computeMemorySize(1.1));
    }

    @Test
    void testComputeMemorySizeFailForNegativeFraction() {
        Assertions.assertThatExceptionOfType(IllegalArgumentException.class).isThrownBy(() -> this.memoryManager.computeMemorySize(-0.1));
    }

    @Test
    void testVerifyEmptyCanBeDoneAfterShutdown() throws MemoryAllocationException, MemoryReservationException {
        this.memoryManager.release((Collection)this.memoryManager.allocatePages(new Object(), 1));
        Object owner = new Object();
        this.memoryManager.reserveMemory(owner, 32768L);
        this.memoryManager.releaseMemory(owner, 32768L);
        this.memoryManager.shutdown();
        this.memoryManager.verifyEmpty();
    }

    private void testCannotAllocateAnymore(Object owner, int numPages) {
        ((ThrowableTypeAssert)Assertions.assertThatExceptionOfType(MemoryAllocationException.class).as("Expected MemoryAllocationException. We should not be able to allocate after allocating or(and) reserving all memory of a certain type.", new Object[0])).isThrownBy(() -> this.memoryManager.allocatePages(owner, numPages));
    }

    private void testCannotReserveAnymore(long size) {
        ((ThrowableTypeAssert)Assertions.assertThatExceptionOfType(MemoryReservationException.class).as("Expected MemoryAllocationException. We should not be able to any more memory after allocating or(and) reserving all memory of a certain type.", new Object[0])).isThrownBy(() -> this.memoryManager.reserveMemory(new Object(), size));
    }
}

