/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.blocklist;

import java.time.Duration;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.flink.runtime.blocklist.BlockedNode;
import org.apache.flink.runtime.blocklist.BlockedNodeAdditionResult;
import org.apache.flink.runtime.blocklist.BlocklistContext;
import org.apache.flink.runtime.blocklist.BlocklistHandler;
import org.apache.flink.runtime.blocklist.BlocklistListener;
import org.apache.flink.runtime.blocklist.BlocklistTracker;
import org.apache.flink.runtime.blocklist.DefaultBlocklistTracker;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.concurrent.ComponentMainThreadExecutor;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;

public class DefaultBlocklistHandler
implements BlocklistHandler,
AutoCloseable {
    private final Logger log;
    private final Function<ResourceID, String> taskManagerNodeIdRetriever;
    private final BlocklistTracker blocklistTracker;
    private final BlocklistContext blocklistContext;
    private final Set<BlocklistListener> blocklistListeners = new HashSet<BlocklistListener>();
    private final Duration timeoutCheckInterval;
    private volatile ScheduledFuture<?> timeoutCheckFuture;
    private final ComponentMainThreadExecutor mainThreadExecutor;

    DefaultBlocklistHandler(BlocklistTracker blocklistTracker, BlocklistContext blocklistContext, Function<ResourceID, String> taskManagerNodeIdRetriever, Duration timeoutCheckInterval, ComponentMainThreadExecutor mainThreadExecutor, Logger log) {
        this.blocklistTracker = (BlocklistTracker)Preconditions.checkNotNull((Object)blocklistTracker);
        this.blocklistContext = (BlocklistContext)Preconditions.checkNotNull((Object)blocklistContext);
        this.taskManagerNodeIdRetriever = (Function)Preconditions.checkNotNull(taskManagerNodeIdRetriever);
        this.timeoutCheckInterval = (Duration)Preconditions.checkNotNull((Object)timeoutCheckInterval);
        this.mainThreadExecutor = (ComponentMainThreadExecutor)Preconditions.checkNotNull((Object)mainThreadExecutor);
        this.log = (Logger)Preconditions.checkNotNull((Object)log);
        this.scheduleTimeoutCheck();
    }

    private void scheduleTimeoutCheck() {
        this.timeoutCheckFuture = this.mainThreadExecutor.schedule(() -> {
            this.removeTimeoutNodes();
            this.scheduleTimeoutCheck();
        }, this.timeoutCheckInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void removeTimeoutNodes() {
        this.assertRunningInMainThread();
        Collection<BlockedNode> removedNodes = this.blocklistTracker.removeTimeoutNodes(System.currentTimeMillis());
        if (!removedNodes.isEmpty()) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Remove {} timeout blocked nodes, details {}. Total {} blocked nodes currently, details: {}.", new Object[]{removedNodes.size(), removedNodes, this.blocklistTracker.getAllBlockedNodes().size(), this.blocklistTracker.getAllBlockedNodes()});
            } else {
                this.log.info("Remove {} timeout blocked nodes. Total {} blocked nodes currently.", (Object)removedNodes.size(), (Object)this.blocklistTracker.getAllBlockedNodes().size());
            }
            this.blocklistContext.unblockResources(removedNodes);
        }
    }

    private void assertRunningInMainThread() {
        this.mainThreadExecutor.assertRunningInMainThread();
    }

    @Override
    public void addNewBlockedNodes(Collection<BlockedNode> newNodes) {
        this.assertRunningInMainThread();
        if (newNodes.isEmpty()) {
            return;
        }
        BlockedNodeAdditionResult result = this.blocklistTracker.addNewBlockedNodes(newNodes);
        Collection<BlockedNode> newlyAddedNodes = result.getNewlyAddedNodes();
        Collection allNodes = Stream.concat(newlyAddedNodes.stream(), result.getMergedNodes().stream()).collect(Collectors.toList());
        if (!newlyAddedNodes.isEmpty()) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Newly added {} blocked nodes, details: {}. Total {} blocked nodes currently, details: {}.", new Object[]{newlyAddedNodes.size(), newlyAddedNodes, this.blocklistTracker.getAllBlockedNodes().size(), this.blocklistTracker.getAllBlockedNodes()});
            } else {
                this.log.info("Newly added {} blocked nodes. Total {} blocked nodes currently.", (Object)newlyAddedNodes.size(), (Object)this.blocklistTracker.getAllBlockedNodes().size());
            }
            this.blocklistListeners.forEach(listener -> listener.notifyNewBlockedNodes(allNodes));
            this.blocklistContext.blockResources(newlyAddedNodes);
        } else if (!allNodes.isEmpty()) {
            this.blocklistListeners.forEach(listener -> listener.notifyNewBlockedNodes(allNodes));
        }
    }

    @Override
    public boolean isBlockedTaskManager(ResourceID taskManagerId) {
        this.assertRunningInMainThread();
        String nodeId = (String)Preconditions.checkNotNull((Object)this.taskManagerNodeIdRetriever.apply(taskManagerId));
        return this.blocklistTracker.isBlockedNode(nodeId);
    }

    @Override
    public Set<String> getAllBlockedNodeIds() {
        this.assertRunningInMainThread();
        return this.blocklistTracker.getAllBlockedNodeIds();
    }

    @Override
    public void registerBlocklistListener(BlocklistListener blocklistListener) {
        this.assertRunningInMainThread();
        Preconditions.checkNotNull((Object)blocklistListener);
        if (!this.blocklistListeners.contains(blocklistListener)) {
            this.blocklistListeners.add(blocklistListener);
            Collection<BlockedNode> allBlockedNodes = this.blocklistTracker.getAllBlockedNodes();
            if (!allBlockedNodes.isEmpty()) {
                blocklistListener.notifyNewBlockedNodes(allBlockedNodes);
            }
        }
    }

    @Override
    public void deregisterBlocklistListener(BlocklistListener blocklistListener) {
        this.assertRunningInMainThread();
        Preconditions.checkNotNull((Object)blocklistListener);
        this.blocklistListeners.remove(blocklistListener);
    }

    @Override
    public void close() throws Exception {
        if (this.timeoutCheckFuture != null) {
            this.timeoutCheckFuture.cancel(false);
        }
    }

    public static class Factory
    implements BlocklistHandler.Factory {
        private final Duration timeoutCheckInterval;

        public Factory(Duration timeoutCheckInterval) {
            this.timeoutCheckInterval = (Duration)Preconditions.checkNotNull((Object)timeoutCheckInterval);
        }

        @Override
        public BlocklistHandler create(BlocklistContext blocklistContext, Function<ResourceID, String> taskManagerNodeIdRetriever, ComponentMainThreadExecutor mainThreadExecutor, Logger log) {
            return new DefaultBlocklistHandler(new DefaultBlocklistTracker(), blocklistContext, taskManagerNodeIdRetriever, this.timeoutCheckInterval, mainThreadExecutor, log);
        }
    }
}

