package org.infinispan.factories;

import org.infinispan.config.ConfigurationException;
import org.infinispan.executors.ExecutorFactory;
import org.infinispan.executors.ScheduledExecutorFactory;
import org.infinispan.factories.annotations.DefaultFactoryFor;
import org.infinispan.util.Util;

import java.util.Properties;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import static org.infinispan.factories.KnownComponentNames.*;

/**
 * A factory that specifically knows how to create named executors.
 *
 * @author Manik Surtani
 * @since 4.0
 */
@DefaultFactoryFor(classes = {ExecutorService.class, Executor.class, ScheduledExecutorService.class})
public class NamedExecutorsFactory extends NamedComponentFactory implements AutoInstantiableFactory {

   private ExecutorService notificationExecutor;
   private ExecutorService asyncTransportExecutor;
   private ScheduledExecutorService evictionExecutor;
   private ScheduledExecutorService asyncReplicationExecutor;

   @SuppressWarnings("unchecked")
   public <T> T construct(Class<T> componentType, String componentName) {
      try {
         // Construction happens only on startup of either CacheManager, or Cache, so
         // using synchronized protection does not have a great impact on app performance.
         if (componentName.equals(ASYNC_NOTIFICATION_EXECUTOR)) {
            synchronized (this) {
               if (notificationExecutor == null) {
                  notificationExecutor = buildAndConfigureExecutorService(
                        globalConfiguration.getAsyncListenerExecutorFactoryClass(),
                        globalConfiguration.getAsyncListenerExecutorProperties(), componentName);
               }
            }
            return (T) notificationExecutor;
         } else if (componentName.equals(ASYNC_TRANSPORT_EXECUTOR)) {
            synchronized (this) {
               if (asyncTransportExecutor == null) {
                  asyncTransportExecutor = buildAndConfigureExecutorService(
                        globalConfiguration.getAsyncTransportExecutorFactoryClass(),
                        globalConfiguration.getAsyncTransportExecutorProperties(), componentName);
               }
            }
            return (T) asyncTransportExecutor;
         } else if (componentName.equals(EVICTION_SCHEDULED_EXECUTOR)) {
            synchronized (this) {
               if (evictionExecutor == null) {
                  evictionExecutor = buildAndConfigureScheduledExecutorService(
                        globalConfiguration.getEvictionScheduledExecutorFactoryClass(),
                        globalConfiguration.getEvictionScheduledExecutorProperties(), componentName);
               }
            }
            return (T) evictionExecutor;
         } else if (componentName.equals(ASYNC_REPLICATION_QUEUE_EXECUTOR)) {
            synchronized (this) {
               if (asyncReplicationExecutor == null) {
                  asyncReplicationExecutor = buildAndConfigureScheduledExecutorService(
                        globalConfiguration.getReplicationQueueScheduledExecutorFactoryClass(),
                        globalConfiguration.getReplicationQueueScheduledExecutorProperties(), componentName);
               }
            }
            return (T) asyncReplicationExecutor;
         } else {
            throw new ConfigurationException("Unknown named executor " + componentName);
         }
      } catch (ConfigurationException ce) {
         throw ce;
      } catch (Exception e) {
         throw new ConfigurationException("Unable to instantiate ExecutorFactory for named component " + componentName, e);
      }
   }

   private ExecutorService buildAndConfigureExecutorService(String factoryName, Properties p, String componentName) throws Exception {
      Properties props = new Properties(p); // defensive copy
      if (p != null && !p.isEmpty()) props.putAll(p);
      ExecutorFactory f = (ExecutorFactory) Util.getInstance(factoryName);
      setComponentName(componentName, props);
      setDefaultThreads(KnownComponentNames.getDefaultThreads(componentName), props);
      setDefaultThreadPrio(KnownComponentNames.getDefaultThreadPrio(componentName), props);
      return f.getExecutor(props);
   }

   private ScheduledExecutorService buildAndConfigureScheduledExecutorService(String factoryName, Properties p, String componentName) throws Exception {
      Properties props = new Properties(); // defensive copy
      if (p != null && !p.isEmpty()) props.putAll(p);
      ScheduledExecutorFactory f = (ScheduledExecutorFactory) Util.getInstance(factoryName);
      setComponentName(componentName, props);
      setDefaultThreadPrio(KnownComponentNames.getDefaultThreadPrio(componentName), props);
      return f.getScheduledExecutor(props);
   }

   private void setDefaultThreadPrio(int prio, Properties props) {
      if (!props.containsKey("threadPriority")) props.setProperty("threadPriority", String.valueOf(prio));
   }

   private void setDefaultThreads(int numThreads, Properties props) {
      if (!props.containsKey("maxThreads")) props.setProperty("maxThreads", String.valueOf(numThreads));
   }

   private void setComponentName(String cn, Properties p) {
      if (cn != null) p.setProperty("componentName", format(cn));
   }

   private String format(String cn) {
      int dotIndex = cn.lastIndexOf(".");
      int dotIndexPlusOne = dotIndex + 1;
      String cname = cn;
      if (dotIndexPlusOne == cn.length())
         cname = format(cn.substring(0, cn.length() - 1));
      else {
         if (dotIndex > -1 && cn.length() > dotIndexPlusOne) {
            cname = cn.substring(dotIndexPlusOne);
         }
         cname += "-thread";
      }
      return cname;
   }
}
